//=============================================================================
// Drivable.
//=============================================================================
// A drivable brush can be controlled by means of a VehicleControl Pickup item.
// The link:  the Tag of the drivable equals the ControlledTag of the pickup.
// A drivable brush can have other actors attached (Child's Event Tag equals drivable's ChildTag),
// including other drivables, dependent or independent.
class Drivable expands Mover;

enum AimingMethod {
   Manual,                // player directly changes rotation
   Stabilized,            // player directly changes desired rotation
   InstantHit,            // rotates to point at player view target 
   BallisticHigh,         // ballistic solution to player view target, high angle(shoot over buildings etc.)
   BallisticLow           // ballistic solution to player view target, low angle (direct fire with real projectiles)
};

enum DrivablePhysicsMethod {
   StaticMover,           // Drivable is not supposed to move - no physics functions are called.
   FloatAndBlock,         // Drivable not influenced by floor or gravity at all. Can use this for turrets and boats.
   OrientToFloor,         // normal pawn beheviour: what is the orientation of the part of the floor I'm on.
   TerrainFollowing,      // same as OrientToFloor, but with more detail. NOT DONE YET
   ForceCalculation       // full-featured calculation - will handle rotation and translation as result of collisions. NOT DONE YET
};

var() bool bHUD_ShowsPitch;
var() bool bHUD_ShowsYaw;
var() bool bHUD_ShowsRoll;
var() bool bHUD_ShowsSpeed;

// Tells drivable to report its location/rotation to the vehiclecontrol controlling it
var() bool bBot_ShowVehicleLocation;
var() bool bBot_ShowTurretLocation;

// Attack parameters
var() float ProjectileSpeed; // although standard drivable doesn't fire projectiles, this is necessary for ballistics
var() AimingMethod Aiming;
var float passedaimingdelay;
var() bool bDoesBallisticCalculation;
var bool PreviousbFire;
var bool PreviousbAltFire;

// Defense parameters
var() int Armor; // armor strength
var() int InitialHitPoints;
var() int FullHitPoints;
var int HitPoints;
var() class<effects> DestructionEffect;
var() bool bHideOnDestruction;

// Movement parameters
var() DrivablePhysicsMethod PhysicsType;
var() bool PhysicsFlatBottom; // modifies determination of swivel points. set to false for wheeled vehicle, true for tracked/hovercraft
var() bool PhysicsHover;      // modifies behaviour of terrainfollowing physics models to make a vehicle hover over water
var() rotator PhysicalRotationRate;      // How fast will the drivable rotate in the world (OrientToFloor physics)
var() float WaterTraction; // should be between 0 and 1, determines how much traction and control the drivable has in the water
var() float AirTraction;   // ditto for air.
var() bool bRestrictPitch; // this, combined with ChildTag, can be used to great effect, e.g. you can create a gun with separate drivables for the yaw and pitch
var() float MinPitch;      // how far down can the vehicle point?
var() float MaxPitch;      // how high
var() float PitchUpRate;   // Positive G's if you will
var() float PitchDownRate; // Negative G's
var() bool bRestrictRoll;  // 
var() float MinRoll;       // how far can we roll to the left
var() float MaxRoll;       // how far can we roll to the right
var() float RollLeftRate;  // roll-rate. Some propellor driven aircraft have asymmetric roll-rates
var() float RollRightRate;
var() bool bRestrictYaw;   
var() float MinYaw;        // How far can we turn (not roll) to the left
var() float MaxYaw;        // How far to the right
var() float YawLeftRate;   // turn rate to the left
var() float YawRightRate;  // to the right

var() sound StationarySound;
var() sound OperatingSound;
var() sound CrashingSound;
var() sound DestroyedSound;

var bool bAimPlayerTarget;  // for simplified control: always attempt to point in the direction of the point the player is looking at
var() float ControlFactorTurretTurn;
var() float ControlFactorSpeed;
var() float ControlFactorTurretUp;
var() float ControlFactorVehicleUp;
var() float ControlFactorVehicleTurn;
var() float Elasticity;    // how well this drivable will bounce back off walls
var drivable parentdrv;       // brush from which we get our rotation and position offsets
var mover PassengerOf;     // brush we're standing on when we're a passenger
var bool bClamped;         // not allowed to move or rotate. That is determined by our base
var bool bRepairing;

// speed control is as follows:
// player indicates desired speed using his forward/backward control;
// vehicle accelerates/decelerates proportionally to the difference, but acceleration/deceleration will
// never exceed the maximum values
var() float MaxSpeed;   // absolute maximum speed of the vehicle. Actual speed is controllable
var() float MinSpeed;   // absolute maximum speed of the vehicle. Actual speed is controllable
var float CurrentSpeed;  
var float DesiredSpeed;
var() float MaxAcceleration; // Maximum value: actual forward acceleration is controllable
var() float MaxDeceleration; // Maximum value: actual deceleration is controllable
var() float MaxReverseAcceleration; // Maximum reverse acceleration 
var bool bPhysics;       // whether or not to apply physics to this drivable - static components don't need physics
var float currenttraction;
var() bool bDestroyed;   // Set to false in editor to have vehicle destroyed at start, so a repair kit is needed
var() bool bRepairable;
var() float RepairTime;  // How long it takes to repair this vehicle

// Control parameters
var bool bStabilized;          // this drivable's rotation is completely independent from its parentdrv - set from aimingmethod
var() bool bIndependent;       // Control input is not propagated from the parentdrv to this drivable
var() bool bCriticalComponent; // if this drivable is destroyed, so is its parentdrv
var bool CtlbJump,CtlbDuck,CtlbFire,CtlbAltFire; // memory to check for 'just fired' etc.

// Misc
var int Health;     // when this hits 0, drivable will be destroyed
var() name ChildTag[20];  // links to other actors that have to move with the drivable. Turrets etc. (use Tag of those actors)
var() name FeedbackTag;
var pawn controllingplayer; // set by controller, used to identify player who fires shots
var vehiclecontrol currentcontroller;
var drivable MainDrivable;  // inherited from parent; used to prevent damage to any actor in the same drivable

var rotator angularvelocity;
var vector linearvelocity;
var rotator CurrentRotation;
var rotator PlaneAngles;
var vector  Origin;
var vector  CurrentLocation;
var vector  PreviousLocation;
var rotator  PreviousRotation;
var vector LastGoodLocation;   // used for recovery in TerrainFollowing physics
var rotator LastGoodRotation;  // ditto
var bool    bUseFeedBack;
var rotator PreviousPlaneAngles;
var vector PreviousOrigin;
var float controlinputYaw;   // buffered values: InterpretControls sets these, DoPhysics uses them
var float controlinputPitch;
var float controlinputRoll;
var rotator StabilizedRotation; // if stabilized, indirectly controls CurrentRotation
var rotator offsetrotation;  // if stabilized, set at start to Rotation
var vector gravityinducedvelocity;
var float passedgroundtime;  // allows travelling over small gaps without much danger of sinking in
var zoneinfo PreviousZone;

struct DrivableChildInfo {
 var actor Act;
 var vector OriginOffset;  // used to calculate Origin
 var rotator planeangles;  // angles (Pitch,Yaw and Roll) between plane of parentdrv and plane of this child
 var vector Origin;
 var rotator originangles;
 var float origindistance;
};

// these arrays are used to store starting locations of child components relative to this, the parentdrv
var DrivableChildInfo DrivableChild[50];
var int DrivableChildCount;
var DrivableChildInfo Feedback[40];
var int FeedbackCount;
var int GroundFeedbackIndex;
var bool bControllable;

var float CurrentRotationYaw,CurrentRotationPitch,CurrentRotationRoll;
var float planeanglesYaw,planeanglesPitch,planeanglesRoll;
var float currentlocationX,CurrentLocationY,CurrentLocationZ;
var float originX,originY,OriginZ;

var float ClientX,ClientY,ClientZ,ClientYaw,ClientPitch,ClientRoll;

var bool bChildInitialized; // has the client initialized itself yet?

replication
{
	// Things the server should send to the client.
	reliable if( Role==ROLE_Authority )
		hitpoints,armor,
		passengerof,
        ClientX,ClientY,ClientZ,ClientYaw,ClientPitch,ClientRoll,
		parentdrv,currentcontroller,currentspeed;
}

// unfortunately, this needs to be used to round the drivable's location before applying
// it to its children.
// Unreal seems to quantize the location, not only for replication, but also for display.
// Since this would happen both for the parentdrv and the child, the child would seem to move
// relative to the parentdrv. This is a very ugly effect.
// Note that it is not necessary to quantize the resulting vector - that is done by unreal
// for display, and the children will quantize their origin.
final simulated function vector qtz(vector invect) 
{
 local vector outvect;
 
 outvect.X=int(invect.X);
 outvect.Y=int(invect.Y);
 outvect.Z=int(invect.Z);

 return outvect;
}


final simulated function int sign(float value)
{
 if(value==0)
  return 0;
 if(value<0)
  return -1;
 return 1;
}

final simulated function float normalizeangle(float inangle)
{
 local int divisions;
 
 divisions = sign(inangle)*int(abs(inangle)/32768);
 
 return inangle-divisions*65536;
}
 
native(3) final simulated function float drv_atan2(float Y,float X);

// atan2 determines 360 degree angle (between -pi and Pi, more precisely)
final simulated function float atan2(float Y,float X)
{
 local float tempang;
 
 if(X==0) {
  if(Y<0)
   tempang=-pi/2.0;
  else if(Y>0)
   tempang=pi/2.0;
  else
   tempang=0;
 }
 else if(X<0) {
  tempang=atan(Y/X)+pi; // Third quadrant
 }
 else 
  tempang=atan(Y/X);        // 
   
 if(tempang>pi) 
  tempang-=pi*2.0;
 
 if(tempang<-pi)
  tempang+=pi*2.0;
   
 return tempang;
}

// Possibly useful functions below

// RealWorldLocation determines the position of a point after its origin is rotated.
// this is needed to properly rotate and move a child when its starting rotation is different
// from that of the parentdrv
// (otherwise, the 'rotateby' could just be added to the child's rotation)
final simulated function vector RealWorldLocation(vector oldpoint,rotator rotateby)
{
 local vector temppoint;
 local rotator temprot;
 local float dist;
 local float axisdist;
 local float axisangle;
 
 temppoint=oldpoint;
 dist = VSize(temppoint);
 temppoint = temppoint/dist; // unity vector, requires less calculations
 
// rotate around X-axis (Roll)
 axisangle = atan2(temppoint.Z,temppoint.Y);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.Y));
 temppoint.Z= sin(axisangle-(RotateBy.Roll*pi)/32768.0)*axisdist;
 temppoint.Y= cos(axisangle-(RotateBy.Roll*pi)/32768.0)*axisdist;

 
// rotate around Y-axis (Pitch)
 axisangle = atan2(temppoint.Z,temppoint.X);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.X));
 temppoint.Z= sin(axisangle+(RotateBy.Pitch*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle+(RotateBy.Pitch*pi)/32768.0)*axisdist;

 temprot=rotateby;
 temprot.Roll=0;
 temprot.Pitch=0;
 
// rotate around Z-axis (Yaw) 
 axisangle = atan2(temppoint.Y,temppoint.X);
 axisdist  = sqrt(square(temppoint.Y)+square(temppoint.X));
 temppoint.Y= sin(axisangle+(RotateBy.Yaw*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle+(RotateBy.Yaw*pi)/32768.0)*axisdist;
 
 return vector(rotator(temppoint))*dist; 
}

final simulated function vector VirtualLocation(vector oldpoint,rotator rotateby)
{
 local vector temppoint;
 local rotator temprot;
 local float dist;
 local float axisdist;
 local float axisangle;
 
 temppoint=oldpoint;
 dist = VSize(temppoint);
 temppoint = temppoint/dist; // unity vector, requires less calculations
 
// rotate around Z-axis (Yaw) 
 axisangle = atan2(temppoint.Y,temppoint.X);
 axisdist  = sqrt(square(temppoint.Y)+square(temppoint.X));
 temppoint.Y= sin(axisangle-(RotateBy.Yaw*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle-(RotateBy.Yaw*pi)/32768.0)*axisdist;
 
// rotate around Y-axis (Pitch)
 axisangle = atan2(temppoint.Z,temppoint.X);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.X));
 temppoint.Z= sin(axisangle-(RotateBy.Pitch*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle-(RotateBy.Pitch*pi)/32768.0)*axisdist;

// rotate around X-axis (Roll)
 axisangle = atan2(temppoint.Z,temppoint.Y);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.Y));
 temppoint.Z= sin(axisangle+(RotateBy.Roll*pi)/32768.0)*axisdist;
 temppoint.Y= cos(axisangle+(RotateBy.Roll*pi)/32768.0)*axisdist;
 
 temprot=rotateby;
 temprot.Roll=0;
 temprot.Pitch=0;
 
 return vector(rotator(temppoint))*dist; // add origin and distance again
}


// getplanerotation determines the angles of a plane defined by two vectors, relative to the ground.
// Two vectors are needed, because roll needs to be determined.
final simulated function rotator GetPlaneRotation(vector Xaxis,vector Yaxis)
{
 local rotator temprot;
 local rotator rotat;
 local vector X,Y;

 X=Xaxis/VSize(Xaxis);
 Y=Yaxis/VSize(Yaxis);

 rotat.Yaw=(atan2(X.y,X.x)*32768.0)/pi;
 temprot = rot(0,0,0);
 temprot.Yaw=rotat.Yaw;
 X = VirtualLocation(X,temprot);
 Y = VirtualLocation(Y,temprot);
 rotat.Pitch = (atan2(X.z,X.x)*32768.0)/pi;
 temprot = rot(0,0,0);
 temprot.Pitch=rotat.Pitch;
 X = VirtualLocation(X,temprot);
 Y = VirtualLocation(Y,temprot);
 rotat.Roll = -(atan2(Y.Z,Y.Y)*32768.0)/pi;                  // I know, this looks strange
  
 return rotat;
}

// RealWorldRotation rotates a rotation
final simulated function rotator RealWorldRotation(rotator CurrentRotation,rotator planeangles)
{
 local vector X,Y;
 
 X=RealWorldLocation(vect(1,0,0),CurrentRotation);
 Y=RealWorldLocation(vect(0,1,0),CurrentRotation);
 X=RealWorldLocation(X,planeangles);
 Y=RealWorldLocation(Y,planeangles);
 
 return GetPlaneRotation(X,Y);
}

// VirtualRotation rotates a rotation
final simulated function rotator VirtualRotation(rotator CurrentRotation,rotator planeangles)
{
 local vector X,Y;
 
 X=RealWorldLocation(vect(1,0,0),CurrentRotation);
 Y=RealWorldLocation(vect(0,1,0),CurrentRotation);
 X=VirtualLocation(X,planeangles);
 Y=VirtualLocation(Y,planeangles);
 
 return GetPlaneRotation(X,Y);
}

simulated final function SplitAcceleration(vector offset,vector acceleration,out vector LinearAcceleration,out rotator angularacceleration)
{
 local vector TempAcceleration;
 local vector ForceOffset;
 local rotator originangles;
 local rotator TempRotation;
 local float origindistance;
 
 TempAcceleration=acceleration;
 originangles = rotator(offset);
 origindistance = vsize(offset);
 
 if(vsize(offset)==0) { // special case: no rotational effect
  LinearAcceleration = TempAcceleration;
  AngularAcceleration = rot(0,0,0);
 }
 else {
  ForceOffset = VirtualLocation(offset,rotator(TempAcceleration));
  LinearAcceleration = abs(Offset.X)/origindistance*TempAcceleration;
  TempAcceleration -= LinearAcceleration; // this much will come from rotation
  // now rotate back to get into our plane
  TempAcceleration = VirtualLocation(TempAcceleration,rotation);
  // rotate the force so our feedbackpoint lies in the X/Y plane (Invert pitch and roll)
  TempRotation = originangles;
  TempRotation.Yaw = 0;
//  TempAcceleration = RealWorldLocation(TempAcceleration,TempRotation);
  TempAcceleration = VirtualLocation(TempAcceleration,TempRotation);
  // originangles.yaw determines how much force is applied to Pitch, and how much to Roll
  // determine roll/pitch effect by looking at Z component of the resulting vector
  AngularAcceleration.Pitch= cos(originangles.yaw/32768*pi)*TempAcceleration.Z;// /feedback[i].origindistance;
  AngularAcceleration.Roll = sin(originangles.yaw/32768*pi)*TempAcceleration.Z;// /feedback[i].origindistance;
  // Now rotate all the way so we can determine what the yaw effect will be
  TempAcceleration = VirtualLocation(TempAcceleration,TempRotation);
  TempAcceleration = VirtualLocation(TempAcceleration,originangles);
  AngularAcceleration.Yaw  = TempAcceleration.Y/origindistance/pi*32768;
 } 
}

simulated final function bool CheckForCollision(float deltatime)
{
 local vector O;
 local vector N;
 local vector curloc,prevloc;
 local rotator currot,prevrot;
 local vector HitNormal;
 local vector HitLocation;
 local DrivableFeedback fb;
 local rotator MovementDirection;
 local int i;

 MovementDirection = rotator(CurrentLocation-PreviousLocation);
 prevloc=RealWorldLocation(PreviousLocation,previousplaneangles)+previousOrigin;
 prevrot=RealWorldRotation(PreviousRotation,previousplaneangles);
 curloc= RealWorldLocation(CurrentLocation,planeangles)+Origin;
 currot= RealWorldRotation(CurrentRotation,planeangles);

 for(i=0;i<FeedbackCount;i++) { // check all of our DrivableFeedback points for collisions
  O=CalculateChildLocation(Feedback[i],prevloc,prevrot);
  N=CalculateChildLocation(Feedback[i],curloc,currot);
  fb = DrivableFeedback(Feedback[i].Act);
  if(!fb.bIsGroundFeedback && !fb.bIsAuxiliaryFeedback) {
   // next test determines if feedbackpoint points roughly in the same direction as the movement is
   if(VirtualLocation(vector(CalculateChildRotation(Feedback[i],curloc,currot)),MovementDirection).X>0) {
    if(Trace(HitLocation,HitNormal,N,O, false)!=None)
     return true;
   }
  }
 }
 return false;
}

simulated final function ControlVehicle(bool bJustFired,bool bFire,bool bJustAltFired,bool bAltFire,
                              float aVehicleSpeedControl,float aTurretTurn,float aVehicleTurn,
                              float aVehicleUp,float aTurretUp,
                              bool bJustJumped, bool bJustDucked)
{
 local Actor Act;
 local Drivable Drv;

 local int i;

 if(IsInState('Activated')) { 
 
  InterpretControls(bJustFired, bFire, bJustAltFired, bAltFire, aVehicleSpeedControl*ControlFactorSpeed,
                    aTurretTurn*ControlFactorTurretTurn, aVehicleTurn*ControlFactorVehicleTurn,
                    aVehicleUp*ControlFactorVehicleUp, aTurretUp*ControlFactorTurretUp, bJustJumped,bJustDucked);

  // propagate movement commands to children
  for(i=0;i<DrivableChildCount;i++) {
   Drv=Drivable(drivableChild[i].Act);
   if(Drv!=None)
    if(!Drv.bIndependent)
     Drv.ControlVehicle(bJustFired, bFire, bJustAltFired, bAltFire,aVehicleSpeedControl, aTurretTurn,
                        aVehicleTurn, aVehicleUp, aTurretUp, bJustJumped,bJustDucked);
  }
 }
}


// moves all children according to current rotation/position and their offsets
final simulated function UpdateAll(float DeltaTime)
{
 local Actor Act;
 local Drivable Drv;
 local int i;
 local rotator newrot;
 local vector X,Y;
 local vector newloc;

 PreviousPlaneangles=planeangles;
 PreviousOrigin=origin;
  
 PreviousZone = region.zone;
  
 if(Role==ROLE_Authority) {
  RotationPhysics(DeltaTime);
  DoPhysics(DeltaTime);
  UpdateOwnLocation();
  newloc=qtz(CalculateOwnLocation());
  newrot=CalculateOwnRotation();
  ClientX=newloc.X;
  ClientY=newloc.Y;
  ClientZ=newloc.Z;
  ClientYaw=newrot.Yaw;
  ClientPitch=newrot.Pitch;
  ClientRoll=newrot.Roll;
  for(i=0;i<FeedbackCount;i++) {
   MoveChildOrigin(Feedback[i],newloc,newrot);
  }
 }
 else {
  newloc.X=ClientX;
  newloc.Y=ClientY;
  newloc.Z=ClientZ;
  newrot.Yaw=ClientYaw;
  newrot.Pitch=ClientPitch;
  newrot.Roll=ClientRoll;
//  SetLocation(qtz(newloc));
  movesmooth(qtz(newloc)-location);
  SetRotation(newrot);
 }
 for(i=0;i<DrivableChildCount;i++) {
  MoveChildOrigin(DrivableChild[i],qtz(newloc),newrot);
  if(Drivable(drivableChild[i].Act)!=None) 
   Drivable(DrivableChild[i].Act).UpdateAll(DeltaTime);
 }
 PreviousPlaneAngles = planeangles;
}

simulated final function bool parentdrvOKToMove(vector Origin,rotator planeangles)
{
 local Drivable Drv;
 local int i;

 if(!OKToMove(origin,planeangles))
  return false;

 for(i=0;i<DrivableChildCount;i++) {
  if(!parentdrvOKToMove(CalculateChildLocation(DrivableChild[i],RealWorldLocation(CurrentLocation,planeangles)+Origin,RealWorldRotation(CurrentRotation,planeangles)),
                     CalculateChildRotation(DrivableChild[i],RealWorldLocation(CurrentLocation,planeangles)+Origin,RealWorldRotation(CurrentRotation,planeangles))))
   return false;
 }
 return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////// Overridable functions below ////////////////////////////////////////

function Touch(Actor Other)
{
 local vector relativevelocity;

return;
/////////////////////////////////// this stuff currently hurts the guys in the vehicle as well
 if(VSize(velocity)>100) {
  relativevelocity=VirtualLocation(velocity-other.velocity,rotator(velocity));
 
  if(relativevelocity.X>600)
   other.takedamage(relativevelocity.X-600,instigator,location,velocity,'Crushed');
 }
}

// function can be called recursively
simulated function bool OKToMove(vector Origin,rotator planeangles)
{

// put code in here to prevent movement under certain conditions
 return true;
}

simulated function rotator CalculateOwnRotation()
{
 return RealWorldRotation(CurrentRotation,planeangles);
}

simulated function vector CalculateOwnLocation()
{
 return RealWorldLocation(CurrentLocation,planeangles)+Origin;
}

simulated function UpdateOwnLocation()
{
 local vector realloc;
 local rotator realrot;
 
 if(parentdrv==None) {
  if(PassengerOf!=None && Role==ROLE_Authority) {
   Origin=PassengerOf.Location;
   planeangles=PassengerOf.Rotation;
  }
 } 
 realloc=qtz(CalculateOwnLocation());
 realrot=CalculateOwnRotation();
 
 DesiredRotation=RealRot;
 bRotateToDesired=true;
 RotationRate=(realrot-rotation)*1000;
// SetRotation(RealRot);
 if(Role==ROLE_Authority) {
  MoveSmooth(RealLoc-Location);
  if(parentdrv==None)
   DetermineBaseChange();
 }
 else
  MoveSmooth(RealLoc-Location);
//  SetLocation(RealLoc);
}

event TakeDamage( int Damage, Pawn EventInstigator, vector HitLocation, vector Momentum, name DamageType)
{
 if(hitpoints>0) {
  if(Damage>Armor) {
   hitpoints-=damage-armor;
   if(hitpoints<=0) {
    if(destructioneffect!=None) {
     Spawn(destructioneffect);
    }
    GotoState('Crashing');
   }    
  }
 }
}

function bool IsDriver(actor Other)
{
 local drivable Drv;
 local int i;
 
 if(currentcontroller.owner==other)
  return true;
 
 for(i=0;i<drivablechildcount;i++) {
  Drv=drivable(drivablechild[i].act);
  if(Drv!=None)
   if(Drv.IsDriver(Other))
    return true;
 }
 return false; 
}

// Return true to abort, false to continue.
function bool EncroachingOn( actor Other )
{
	if ( Drivable(Other.Base)!=None && Drivable(Other.Base).MainDrivable==MainDrivable)
	 return false;
	
	if(RealWorldLocation(Velocity-Other.Velocity,rotator(other.location-location)).X<50) {
//	 other.velocity=velocity;
	 return false;
	}

//    other.velocity=velocity;

    if(IsDriver(Other))
     return false;
     	
	return Super.EncroachingOn(Other);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
// These are pretty complex functions, but if you would ever want to store more info for a child
// than 'Drivable' stores, override these 5 functions

simulated function vector CalculateChildLocation(drivablechildinfo child,
                                       vector ownlocation,rotator ownrotation)
{
 return RealWorldLocation(child.originoffset,OwnRotation)+OwnLocation;
}

simulated function rotator CalculateChildViewRotation(drivablechildinfo child,
                                       vector ownlocation,rotator ownrotation)
{
 return RealWorldRotation(child.planeangles,OwnRotation);
}

simulated function rotator CalculateChildRotation(drivablechildinfo child,
                                       vector ownlocation,rotator ownrotation)
{
 return RealWorldRotation(child.planeangles,OwnRotation);
}

simulated function StoreDrivableChildInfo(out DrivableChildInfo dest,Actor Act)
{
 local drivable drv;
 
 dest.Act = Act;
 
 dest.planeangles  = VirtualRotation(Act.Rotation,Rotation);
 
 dest.originoffset = VirtualLocation(Act.Location-Location,Rotation); 
 dest.Origin = Act.Location;
 dest.origindistance = vsize(dest.originoffset);     // These two are used often and don't change 
 dest.originangles   = rotator(dest.originoffset);   // so being cached here

 drv = drivable(dest.act);
 if(drv!=None) {
  drv.Origin = dest.Origin;
  drv.planeangles = dest.planeangles;
  drv.CurrentRotation = rot(0,0,0);
  drv.CurrentLocation = vect(0,0,0);
 }
}

// this function moves the child component so that it keeps its relative position and rotation
simulated function MoveChildOrigin(drivablechildinfo child,vector baseloc,rotator baserot)
{
 local vector ChildOrigin;
 local rotator NewChildRotation;
 local rotator ChildRelativeRotation;
 local vector O;
 local vector parentdrvO;
 local drivable Drv;
 local vehiclecontrol vc;
 local DrivableDecoration deco;

// calculate child's origin in real world coordinates 
 O=Child.originoffset;             // where is the child's origin in our own space
 O=qtz(CalculateChildLocation(Child,baseloc,baserot));

 Child.act.velocity=velocity; 
 Drv=Drivable(Child.Act);
 if(Drv!=None) { // if drivable, child can update its own position
//  if(Role==ROLE_Authority) { // to prevent rounding errors, simply let the server calculate. Price is: child will lag.
   Drv.Origin = O;
   Drv.planeangles=CalculateChildRotation(Child,baseloc,baserot);
//  }
 }
 else { // if not a drivable, do the next best thing: rotate it yourself
  if(DrivableCameraComponent(child.act)!=None || VehicleControl(Child.act)!=None)
   NewChildRotation=CalculateChildViewRotation(Child,baseloc,baserot);
  else
   NewChildRotation=CalculateChildRotation(Child,baseloc,baserot);

  if(DrivableFeedback(child.act)==None) {
   child.Act.SetLocation(O);
   child.Act.SetRotation(NewChildRotation);
   child.Act.SetBase(self);
  }
//  child.Act.velocity = velocity;
  deco=DrivableDecoration(Child.Act);
  if(deco!=None && Role==ROLE_Authority) 
   deco.ReplicateLocation(O,NewChildRotation);
   
  vc=VehicleControl(Child.Act);
  if(vc!=None) {
   vc.CurrentLocation = O;
   vc.CurrentRotation = NewChildRotation;
   vc.UpdateLocation();
  }
 }
}


/////////////////////////////////////////////////////////////////////////////////////////////////
// DoOpen() and DoClose() are empty to defeat keyframe behaviour when something bumps us
function DoOpen()
{

}


function DoClose()
{

}



///////////////////////////////////////////////////////////////////////////////////////////
// States and initialization

// Tick can be overridden to support animation, although that could also be handled in DoPhysics
simulated event Tick(float DeltaTime)
{
 if(Role<ROLE_Authority) { // Rotators get rounded in replication
  currentrotation.yaw=currentrotationyaw;         
  currentrotation.pitch=currentrotationpitch;
  currentrotation.roll=currentrotationroll;
  planeangles.yaw=planeanglesyaw;
  planeangles.pitch=planeanglespitch;
  planeangles.roll=planeanglesroll;
  currentlocation.X=currentlocationX;
  currentlocation.Y=currentlocationY;
  currentlocation.Z=currentlocationZ;
  origin.X=originX;
  origin.Y=originY;
  origin.Z=originZ;
 }
 
 if(parentdrv==none) 
  UpdateAll(deltatime);
 if(currentcontroller!=None) {
  // this code executed both on the server and the client. Maybe a bot might find this useful.
  if(bHUD_showspitch)
   currentcontroller.reportedrotationpitch=180.0*rotation.pitch/32768.0;
  if(bHUD_showsyaw)
   currentcontroller.reportedrotationyaw=180.0*rotation.yaw/32768.0;
  if(bHUD_showsroll)
   currentcontroller.reportedrotationroll=180.0*rotation.roll/32768.0;
  if(bHUD_showsspeed)
   currentcontroller.reportedspeed=currentspeed*2.54*3600.0/120000.0; // speed in kph (provided Unreal time units are seconds)
  if(bBot_ShowVehicleLocation) {
   currentcontroller.reportedvehiclelocation=location;
   currentcontroller.reportedvehiclerotation=rotation;
  }
  if(bBot_ShowTurretLocation) {  
   currentcontroller.reportedturretrotation=rotation;
   currentcontroller.reportedturretlocation=location;
  }

  if((parentdrv==None || bIndependent) && Role==ROLE_Authority && !bDestroyed) {

   ControlVehicle(currentcontroller.CtlbFire && !CtlbFire,currentcontroller.CtlbFire,
                  currentcontroller.CtlbAltFire && !CtlbAltFire,currentcontroller.CtlbAltFire,
                  currentcontroller.CtlaVehicleSpeedControl,currentcontroller.CtlaTurretTurn,
                  currentcontroller.CtlaVehicleTurn,currentcontroller.CtlaVehicleUp,
                  currentcontroller.CtlaTurretUp,
                  currentcontroller.CtlaVehicleUp>0.0 && !CtlbJump,
                  currentcontroller.CtlaVehicleUp<0.0 && !CtlbDuck);

   CtlbFire=currentcontroller.CtlbFire;
   CtlbAltFire=currentcontroller.CtlbAltFire;
   if(currentcontroller.CtlaVehicleUp>0.0)
    CtlbJump=true;
   else
    CtlbJump=false;

   if(currentcontroller.CtlaVehicleUp<0.0)
    CtlbDuck=true;
   else
    CtlbDuck=false;
  }
 }  
 if(Role==ROLE_Authority) {
  currentrotationyaw=currentrotation.yaw;
  currentrotationpitch=currentrotation.pitch;
  currentrotationroll=currentrotation.roll;
  planeanglesyaw=planeangles.yaw;
  planeanglespitch=planeangles.pitch;
  planeanglesroll=planeangles.roll;
  currentlocationX=currentlocation.X;
  currentlocationY=currentlocation.Y;
  currentlocationZ=currentlocation.Z;
  originX=origin.X;
  originY=origin.Y;
  originZ=origin.Z;
 }
}

// Override these two if you don't want to be bothered with overriding the whole Activated state
// You can use these functions, for example to have a turret slide in and out of its casing like the
// autoturrets in Assault maps
function Activation()
{

}

function Deactivation()
{

}

// When destroyed, the vehicle is completely dead - no input, no physics
state DestroyedState
{
 ignores Tick;          // no physics
 ignores TakeDamage;    // no additional damage
 
 function BeginState()
 {
  bDestroyed=true;
  AmbientSound=DestroyedSound;
 }
}

// Use this state to let the vehicle collapse or fall down or sink etc.
state Crashing
{
 ignores TakeDamage;

 function BeginState()
 {
  local int i;
  local Drivable Drv;
  local vehiclecontrol control;

  bDestroyed=true;
  AmbientSound=CrashingSound;
  
  if(bCriticalComponent && !bIndependent && parentdrv!=None && !parentdrv.bDestroyed)
   parentdrv.GotoState('Crashing');
  
  for(i=0;i<DrivableChildCount;i++) {
   Drv=Drivable(drivableChild[i].Act);

   if(Drv!=None && !Drv.bDestroyed)
    Drv.GotoState('Crashing');
  }
 }
 
Begin:  
  Sleep(1);
  gotostate('DestroyedState');
}

// This state delays the actual use of the drivable. It might also be used for repair animation of some kind.
state Repairing
{
 ignores Tick;          // no physics
 ignores TakeDamage;


 function BeginState()
 {
  local int i;
  local Drivable Drv;

  for(i=0;i<DrivableChildCount;i++) {
   Drv=Drivable(drivableChild[i].Act);

   if(Drv!=None && Drv.bRepairable && Drv.bDestroyed && !Drv.bRepairing && !Drv.bIndependent)
    Drv.GotoState('Repairing');
  }
 }
 
 Begin:
  bRepairing=true;
  Sleep(repairtime);
  bDestroyed=false;
  HitPoints=FullHitPoints;
  bRepairing=false;
  GotoState('Deactivated');
}

// When deactivated, there is no input, but the vehicle does abide by the rules of physics
state Deactivated
{
 function BeginState()
 {
  local int i;
  
  AmbientSound=StationarySound;

  controlinputYaw=0;
  controlinputPitch=0;
  controlinputRoll=0;

  controllingplayer = none;
  currentcontroller = none;
  instigator = none;
  desiredspeed = 0;
 }
}

// When activated, both input and physics are applied.
state Activated
{
 function BeginState()
 {
  local int i;
  local Drivable Drv;
  local vehiclecontrol control;


  AmbientSound=OperatingSound;

  controlinputYaw=0;
  controlinputPitch=0;
  controlinputRoll=0;
  PassedAimingDelay=0;
  
  
  if(!bIndependent)
   if(parentdrv!=None) {
    if(parentdrv.currentcontroller!=None) { // only inherit if there is something to inherit
     ControllingPlayer = parentdrv.ControllingPlayer;
     CurrentController = parentdrv.currentcontroller;
    }
   }

  Instigator = ControllingPlayer;
  
  for(i=0;i<DrivableChildCount;i++) {
   Drv=Drivable(drivableChild[i].Act);
   if(Drv!=None)
    if(!Drv.bIndependent)
     Drv.GotoState('Activated');
  }
  Activation();
 }

 function EndState()
 {
  local int i;
  local Drivable Drv;
  local vehiclecontrol control;

 
  DesiredSpeed = 0;
  currentcontroller = None;
  ControllingPlayer=None;

  controlinputYaw=0;
  controlinputPitch=0;
  controlinputRoll=0;

  Deactivation();
   
  for(i=0;i<DrivableChildCount;i++) {
   Drv=Drivable(drivableChild[i].Act);

   if(Drv!=None)
    if(!Drv.bIndependent)
     Drv.GotoState('Deactivated');
  }
 }
}


// Initialization, phase 1
simulated function BeginPlay()
{
 local int i;

 bPhysics=false;
 
 if(PhysicsType!=StaticMover)
   bPhysics=true;
  
 currenttraction=1; // will not change if child
 CurrentController = None;
 ControllingPlayer=None;
 CurrentSpeed=0;
 DesiredSpeed=0;
 groundfeedbackindex=0;
 CurrentRotation = rotation;              // initialize as if parentdrv == None
 parentdrv = none;                        // if later, during PostBeginPlay, parent is set,
 Origin = vect(0,0,0);                    // parent will update these
 MainDrivable=Self;
 CurrentLocation = Location;
 PreviousLocation = CurrentLocation;
 PreviousRotation = CurrentRotation;
 LastGoodLocation = CurrentLocation;
 LastGoodRotation = CurrentRotation;
 planeangles=rot(0,0,0);
 for(i=0;i<ArrayCount(DrivableChild);i++)
  DrivableChild[i].Act=None;
 DrivableChildCount=0;
 for(i=0;i<ArrayCount(Feedback);i++)
  Feedback[i].Act=None;
 FeedbackCount=0;
 Hitpoints=initialhitpoints;
 controlinputYaw=0;
 controlinputPitch=0;
 controlinputRoll=0;
 switch(Aiming) {
  case BallisticLow:
  case BallisticHigh:
  case InstantHit:
   bAimPlayerTarget=true;
   bStabilized=true;
   break;
  case Stabilized:
   bAimPlayerTarget=false;
   bStabilized=true;
   break;
  default:
   bStabilized=false;
   bAimPlayerTarget=false;
   break;
 }
 if(bStabilized)
  OffsetRotation=rotation;
 else
  OffsetRotation=rot(0,0,0);
}

// Immediately after mover enters gameplay.
// Initialization, phase 2
simulated function PostBeginPlay()
{
	local Actor Act;
	local Mover Mov;
	local drivable Drv;
   
	local int i;

//	Super.PostBeginPlay();

	// Now initialize all children that are directly influenced by this drivable
	for(i=0;i<arraycount(ChildTag);i++) {
	 if ( ChildTag[i] != '' ) {
	  foreach AllActors( class 'Actor', Act, ChildTag[i] )
	  {
	   if(Act!=Self && DrivableChildCount<ArrayCount(DrivableChild) ) { // we don't want to move ourselves if their tag is the same as ours (children controlled by the same VehicleControl)
 	    StoreDrivableChildInfo(DrivableChild[DrivableChildCount],Act);
	    DrivableChildCount++;
	   }
	   if(Act!=Self) { // this really shouldn't be necessary, since one would set a controller only for the independent movers
	    Drv = Drivable(Act);
	    if(Drv!=None) {
	     Drv.parentdrv = self;
	    }
	   }
	  }
	 }
	}
    if(FeedbackTag!='') {
     foreach AllActors( class 'Actor', Act, FeedbackTag )
	 {
      if(FeedbackCount<ArrayCount(Feedback) && DrivableFeedback(Act)!=None) {
       StoreDrivableChildInfo(Feedback[FeedbackCount],Act);
       if(DrivableFeedback(Act).bIsGroundFeedback) {
        GroundFeedbackIndex=FeedbackCount;
        Feedback[Feedbackcount].originoffset.X=0;  // make sure the ground feedback is right beneath the center
        Feedback[Feedbackcount].originoffset.Y=0;
       }
       FeedbackCount++;
	  }
	 }
    } 
    previousplaneangles=planeangles;
}

// Initialization, phase 3
// This state takes care of the final setup actions
state auto setup
{
 simulated function BeginState()
 {
  local int i;
  local Actor Act;
  
  if(Role<ROLE_Authority) { // not certain that these are necessary
   BeginPlay();
   PostBeginPlay();
  }
  if(ParentDrv==None)
   InheritStuff();

  PreviousZone = region.zone;
  if(bStabilized && parentdrv!=None)
   StabilizedRotation=Rotation;
  else
   StabilizedRotation = rot(0,0,0);

  if(bDestroyed)
   GotoState('DestroyedState');
  else
   GotoState('Deactivated');
 }
}


function InheritStuff()
{
 local Actor Act;
 local drivable Drv;
   
 local int i;


 for(i=0;i<DrivableChildCount;i++) {
  Drv=Drivable(DrivableChild[i].act);
  if(Drv!=None) {
   Drv.MainDrivable=MainDrivable;
   Drv.InheritStuff();
  }
 }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// CONTROL                                                                                       //
///////////////////////////////////////////////////////////////////////////////////////////////////

// Override InterpretControls to define vehicle controls
// Default is to use aUp for Up/Down, aForward for forward/backward, aStrafe for roll left/right, aTurn for yaw left/right


simulated function InterpretControls(bool bJustFired,bool bFire,bool bJustAltFired,bool bAltFire,float aVehicleSpeedControl,
                           float aTurretTurn,float aVehicleTurn,float aVehicleUp,float aTurretUp,bool bJustJumped,bool bJustDucked)
{
 local float NewDesiredSpeed;

 if(aTurretTurn<0)
  controlinputYaw = (aTurretTurn/16384.0)*YawLeftRate;
 else
  controlinputYaw = (aTurretTurn/16384.0)*YawRightRate;
  
 if(aTurretUp<0)
  controlinputPitch = (aTurretUp/16384.0)*PitchUpRate;
 else
  controlinputPitch = (aTurretUp/16384.0)*PitchDownRate;
  
 if(aVehicleTurn<0)
  controlinputRoll = (aVehicleTurn/16384.0)*RollLeftRate;
 else
  controlinputRoll = (aVehicleTurn/16384.0)*RollRightRate;

 NewDesiredSpeed = 0;
 
 if(aVehicleSpeedControl<0)
  NewDesiredSpeed = MinSpeed/16384.0*abs(aVehicleSpeedControl);
 
 if(aVehicleSpeedControl>0)
  NewDesiredSpeed = MaxSpeed/16384.0*abs(aVehicleSpeedControl);

 DesiredSpeed = NewDesiredSpeed;
}


/////////////////////////////////////////////////////////////////////////////////////////////////
// PHYSICS                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////

// Override DoPhysics to define new vehicle types
simulated function DoPhysics(float DeltaTime)
{
 if(parentdrv!=None)
  return;
  
 if(DeltaTime<=0)
  return;
  
 switch(PhysicsType) {
  case StaticMover:
   return;
  case FloatAndBlock:
   FABPhysics(DeltaTime);
   return;
  case OrientToFloor:
   OTFPhysics(DeltaTime);
   return;
  case TerrainFollowing:
   TFPhysics(DeltaTime);
   return;
  case ForceCalculation:
   ForceCalculationPhysics(DeltaTime);
   return;
 }
}

// This is a pretty complex function, since it includes gun stabilization and ballistics. It is defined
// here, at the base class, because the guns themselves are usually mounted on some other drivable, which
// is the actual stabilized drivable.
simulated function RotationPhysics(float DeltaTime)
{
 local rotator NewRotation;
 local rotator NewStabilizedRotation;
 local float compensationYaw,compensationPitch,compensationRoll; // not using rotator because these values are generally<1
 local vector eyelocation,hitnormal,hitlocation,targetvector;
 local float high_angle,low_angle,flighttime;
 local rotator actualrot;
 
 if(DeltaTime<=0)
  return;
  
 if(!bPhysics)
  return;
   
 PreviousRotation = CurrentRotation;
 CurrentRotation = CurrentRotation+AngularVelocity*deltatime;
 CurrentRotation.Roll=normalizeangle(CurrentRotation.Roll);
 CurrentRotation.Pitch=normalizeangle(CurrentRotation.Pitch);
 CurrentRotation.Yaw=normalizeangle(CurrentRotation.Yaw);
 NewRotation = CurrentRotation;
 
 if(!bStabilized)
  StabilizedRotation = CurrentRotation;
  
 if(currentcontroller != None) {  
  PassedAimingDelay+=deltatime;
  if(bAimPlayerTarget && bStabilized && currentcontroller.bplayercanlook) { // if player can look, aim at his target, otherwise act stabilized
   if(ControllingPlayer != None) {
    if(Aiming==InstantHit || PassedAimingDelay>=0.5) {
     eyelocation=ControllingPlayer.Location+RealWorldLocation(vect(0,0,1)*controllingplayer.BaseEyeHeight,controllingplayer.rotation);
     controllingplayer.Trace(HitLocation,HitNormal,
                             eyelocation + 65536.0*vector(controllingplayer.ViewRotation),
                             eyelocation + vector(controllingplayer.ViewRotation)*controllingplayer.collisionradius, True); 
     PassedAimingDelay=0;
     switch(Aiming) {
      case BallisticHigh:
       targetvector = HitLocation-Location;
       NewStabilizedRotation=rotator(HitLocation-Location);
       HighBallisticSolution(sqrt(targetvector.X*targetvector.X+targetvector.Y*targetvector.Y),targetvector.Z,high_angle,flighttime);
       NewStabilizedRotation.Pitch=high_angle;
       if(high_angle>=-16384)
        StabilizedRotation=NewStabilizedRotation;
       break;
      case BallisticLow:
       targetvector = HitLocation-Location;
       NewStabilizedRotation=rotator(HitLocation-Location);
       DirectFireBallisticSolution(sqrt(targetvector.X*targetvector.X+targetvector.Y*targetvector.Y),targetvector.Z,low_angle);
       NewStabilizedRotation.Pitch=low_angle;
       if(low_angle>=-16384)
        StabilizedRotation=NewStabilizedRotation;
       break;
      case InstantHit:
       StabilizedRotation=rotator(HitLocation-Location);
       break;
     }
    }
   }
  } // if stabilized, aiming and looking all enabled
  else { // stabilized/manual behaviour
  // translate control input into desired rotation for the drivable
   compensationYaw=ControlInputYaw*deltatime;
   compensationPitch=ControlInputPitch*deltatime;
   compensationRoll=ControlInputRoll*deltatime; 

   // combined rotation rate may not exceed maximum values
   if(bStabilized) { // controls may be used to their full extent even if the drivable cannot keep up
    if(compensationYaw<-YawLeftRate*DeltaTime)
     compensationYaw=-YawLeftRate*DeltaTime;
    else
    if(compensationYaw>YawRightRate*DeltaTime)
     compensationYaw=YawRightRate*DeltaTime;
   
    if(compensationPitch<-PitchDownRate*DeltaTime)
     compensationPitch=-PitchDownRate*DeltaTime;
    else
    if(compensationPitch>PitchUpRate*DeltaTime)
     compensationPitch=PitchUpRate*DeltaTime;
   
    if(compensationRoll<-RollLeftRate*DeltaTime)
     compensationRoll=-RollLeftRate*DeltaTime;
    else
    if(compensationRoll>RollRightRate*DeltaTime)
     compensationRoll=RollRightRate*DeltaTime;
   }
   else { // manual control: traction determines how fast we can turn
    if(compensationYaw<-YawLeftRate*DeltaTime*currenttraction)
     compensationYaw=-YawLeftRate*DeltaTime*currenttraction;
    else
    if(compensationYaw>YawRightRate*DeltaTime*currenttraction)
     compensationYaw=YawRightRate*DeltaTime*currenttraction;
  
    if(compensationPitch<-PitchDownRate*DeltaTime*currenttraction)
     compensationPitch=-PitchDownRate*DeltaTime*currenttraction;
    else
    if(compensationPitch>PitchUpRate*DeltaTime*currenttraction)
     compensationPitch=PitchUpRate*DeltaTime*currenttraction;
  
    if(compensationRoll<-RollLeftRate*DeltaTime*currenttraction)
     compensationRoll=-RollLeftRate*DeltaTime*currenttraction;
    else
    if(compensationRoll>RollRightRate*DeltaTime*currenttraction)
     compensationRoll=RollRightRate*DeltaTime*currenttraction;
   }

   if(YawLeftRate!=0 || YawRightRate!=0 || !bStabilized)     // if we have no way to modify an axis, don't assume the player wants to
    StabilizedRotation.Yaw += compensationYaw;
   else
    StabilizedRotation.Yaw = planeangles.Yaw;
   
   if(RollLeftRate!=0 || RollRightRate!=0 || !bStabilized) 
    StabilizedRotation.Roll += compensationRoll;
   else
    StabilizedRotation.Roll = planeangles.Roll;
    
   if(PitchUpRate!=0 || PitchDownRate!=0 || !bStabilized) 
    StabilizedRotation.Pitch += compensationPitch;
   else
    StabilizedRotation.Pitch = planeangles.Pitch;

   StabilizedRotation.Roll = normalizeangle(StabilizedRotation.Roll);  
   StabilizedRotation.Pitch = normalizeangle(StabilizedRotation.Pitch);  
   StabilizedRotation.Yaw = normalizeangle(StabilizedRotation.Yaw);  
  } 
  if(bRestrictPitch) {
   if(StabilizedRotation.Pitch>MaxPitch+OffsetRotation.Pitch)
    StabilizedRotation.Pitch=MaxPitch+OffsetRotation.Pitch;
   if(StabilizedRotation.Pitch<MinPitch+OffsetRotation.Pitch)
    StabilizedRotation.Pitch=MinPitch+OffsetRotation.Pitch;
  }
  if(bRestrictRoll) {
   if(StabilizedRotation.Roll>MaxRoll+OffsetRotation.Roll)
    StabilizedRotation.Roll=MaxRoll+OffsetRotation.Roll;
   if(StabilizedRotation.Roll<MinRoll+OffsetRotation.Roll)
    StabilizedRotation.Roll=MinRoll+OffsetRotation.Roll;
  }
  if(bRestrictYaw) {
   if(StabilizedRotation.Yaw>MaxYaw+OffsetRotation.Yaw)
    StabilizedRotation.Yaw=MaxYaw+OffsetRotation.Yaw;
   if(StabilizedRotation.Yaw<MinYaw+OffsetRotation.Yaw)
    StabilizedRotation.Yaw=MinYaw+OffsetRotation.Yaw;
  }
 }
 if(!bStabilized) {// if not stabilized, we control the drivable manually
  NewRotation = StabilizedRotation;
 }
 else {
  actualrot = VirtualRotation(StabilizedRotation,planeangles);
  compensationYaw=normalizeangle(actualrot.Yaw-CurrentRotation.Yaw);
  compensationPitch=normalizeangle(actualrot.Pitch-CurrentRotation.Pitch);
  compensationRoll=normalizeangle(actualrot.Roll-CurrentRotation.Roll);

  // combined rotation rate may not exceed maximum values
  if(compensationYaw<-YawLeftRate*DeltaTime*currenttraction)
   compensationYaw=-YawLeftRate*DeltaTime*currenttraction;
  else
  if(compensationYaw>YawRightRate*DeltaTime*currenttraction)
   compensationYaw=YawRightRate*DeltaTime*currenttraction;
  
  if(compensationPitch<-PitchDownRate*DeltaTime*currenttraction)
   compensationPitch=-PitchDownRate*DeltaTime*currenttraction;
  else
  if(compensationPitch>PitchUpRate*DeltaTime*currenttraction)
   compensationPitch=PitchUpRate*DeltaTime*currenttraction;
  
  if(compensationRoll<-RollLeftRate*DeltaTime*currenttraction)
   compensationRoll=-RollLeftRate*DeltaTime*currenttraction;
  else
  if(compensationRoll>RollRightRate*DeltaTime*currenttraction)
   compensationRoll=RollRightRate*DeltaTime*currenttraction;

  NewRotation.Roll = normalizeangle(NewRotation.Roll+compensationRoll);
  NewRotation.Pitch = normalizeangle(NewRotation.Pitch+compensationPitch);
  NewRotation.Yaw = normalizeangle(NewRotation.Yaw+compensationYaw);
  if(bRestrictPitch) {
   if(NewRotation.Pitch>MaxPitch+OffsetRotation.Pitch)
    NewRotation.Pitch=MaxPitch+OffsetRotation.Pitch;
   if(NewRotation.Pitch<MinPitch+OffsetRotation.Pitch)
    NewRotation.Pitch=MinPitch+OffsetRotation.Pitch;
  }
  if(bRestrictRoll) {
   if(NewRotation.Roll>MaxRoll+OffsetRotation.Roll)
    NewRotation.Roll=MaxRoll+OffsetRotation.Roll;
   if(NewRotation.Roll<MinRoll+OffsetRotation.Roll)
    NewRotation.Roll=MinRoll+OffsetRotation.Roll;
  }
  if(bRestrictYaw) {
   if(NewRotation.Yaw>MaxYaw+OffsetRotation.Yaw)
    NewRotation.Yaw=MaxYaw+OffsetRotation.Yaw;
   if(NewRotation.Yaw<MinYaw+OffsetRotation.Yaw)
    NewRotation.Yaw=MinYaw+OffsetRotation.Yaw;
  }
 }
  
 CurrentRotation = NewRotation; 
}

// DetermineBaseChange shoots a line of length 1 down from the feedbackpoint with bIsGroundFeedback
// then automatically sets its base if necessary
simulated function DetermineBaseChange()
{
 local vector HitNormal;
 local vector HitLocation;
 local DrivableFeedback fb;
 local rotator MovementDirection;
 local mover potentialbase;
 local vector N,O;

 if(parentdrv!=None)
  return;
  
 if(Feedback[groundfeedbackindex].act==None)
  return;

 MovementDirection=rotator(region.zone.zonegravity);
 fb = DrivableFeedback(Feedback[groundfeedbackindex].Act);
 N=Feedback[groundfeedbackindex].originoffset;             // where is the child's origin in our own space
 N=RealWorldLocation(N,CurrentRotation);// We may have rotated in our own plane
 N=RealWorldLocation(N,planeangles);    // Our own plane may have been rotated
 N+=RealWorldLocation(CurrentLocation,planeangles)+Origin;
 O=Feedback[groundfeedbackindex].originoffset*1.25;             // where is the child's origin in our own space
 O=RealWorldLocation(O,CurrentRotation);// We may have rotated in our own plane
 O=RealWorldLocation(O,planeangles);    // Our own plane may have been rotated
 O+=RealWorldLocation(CurrentLocation,planeangles)+Origin;
 potentialbase=Mover(Trace(HitLocation,HitNormal,O,N, false));
 if(potentialbase!=None) {
  if(PassengerOf==None) {
   PassengerOf=potentialbase;
   CurrentLocation = VirtualLocation(CurrentLocation,passengerof.rotation)-passengerof.location;
   CurrentRotation = VirtualRotation(CurrentRotation,passengerof.rotation);
   Origin=potentialbase.location;
   planeangles=potentialbase.rotation;
  }
 }
 else {
  if(PassengerOf!=None) {
  
   CurrentLocation = RealWorldLocation(CurrentLocation,passengerof.rotation)+passengerof.location;
   CurrentRotation = RealWorldRotation(Currentrotation,passengerof.rotation);
   PassengerOf=None;
   Origin=vect(0,0,0);
   planeangles=rot(0,0,0);
  }
 }
}

// Currently, updatespeed looks at the desired speed and determines the acceleration needed,
// so the player does not influence acceleration directly. Main reason for this is that
// there is currently no friction, so the vehicle would tend to travel at its maximum speed.
// Also, this method resembles normal player movement.
simulated function UpdateSpeed(float DeltaTime,float traction)
{
 local float accel;

  
 accel=0.0;
 if(MaxSpeed>0) {
  if(DesiredSpeed<CurrentSpeed) {
   if(CurrentSpeed>0.0 && MaxDeceleration>MaxReverseAcceleration)
    accel=MAX(DesiredSpeed-CurrentSpeed,-MaxDeceleration*deltatime*traction); // note: desiredspeed-currentspeed is negative, so accel is negative
   else
    accel=MAX(DesiredSpeed-CurrentSpeed,-MaxReverseAcceleration*deltatime*traction); // note: desiredspeed-currentspeed is negative, so accel is negative
  }
  if(DesiredSpeed>CurrentSpeed) {
   if(CurrentSpeed<0.0 && MaxDeceleration>MaxAcceleration)
    accel=MIN(DesiredSpeed-CurrentSpeed,MaxDeceleration*deltatime*traction); 
   else
    accel=MIN(DesiredSpeed-CurrentSpeed,MaxAcceleration*deltatime*traction); 
  }
 }
 CurrentSpeed+=accel;
 if(CurrentSpeed>MaxSpeed*traction)
  CurrentSpeed=MaxSpeed*traction;
 else
 if(CurrentSpeed<MinSpeed*traction)
  CurrentSpeed=MinSpeed*traction;
  
 linearvelocity = CurrentSpeed*vector(CurrentRotation);
}

///////////////////////////////////////////////////////////////////////////////////////////////
// Float and Block physics model                                                             //
// Moves in the X direction (which is indicated by the arrow in Unrealed)                    //
// if a collision occurs, simply stops moving.                                               //
// This is the simplest physics model. Suitable for boats and other 2-dimensional movers.    //
///////////////////////////////////////////////////////////////////////////////////////////////
simulated function FABPhysics(float DeltaTime)
{
 local vector LinearAcceleration,NewLocation;
 local rotator AngularAcceleration;
 local float collisiontime;
 local rotator MovementDirection;
 
 currenttraction=1;
 PreviousLocation = CurrentLocation;

 // calculate speed

 UpdateSpeed(deltatime,1);

 CurrentLocation = PreviousLocation+deltatime*linearvelocity; 

 if(CheckForCollision(deltatime)) {
  CurrentLocation = PreviousLocation;
  CurrentRotation = PreviousRotation;
  linearvelocity = vect(0,0,0);
  CurrentSpeed=0;
 }

 if(vsize(linearvelocity)<2)
  linearvelocity=vect(0,0,0);

 velocity = linearvelocity;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// OrientToFloor physics model.
// Best used with small vehicles (e.g. remote controlled vehicles).
// For large vehicles, it is better to use the TerrainFolling model. This will turn and lift the
// vehicle also when only the corners are on the floor.
// A line is drawn from the center of mass in the direction of the gravity vector.
// Where the line hits the ground (if it does, determined by the size of the gravity vector)
// the drivable will pitch and roll to orient itself flat on the floor.
// There is special handling of the ground feedback in case PhysicsHover is enabled (hovercraft hovers on water)
///////////////////////////////////
simulated function OTFPhysics(float DeltaTime)
{
 local float accel;
 local vector LinearAcceleration,NewLocation;
 local rotator AngularAcceleration;
 local float collisiontime;
 local rotator MovementDirection;
 local vector HitNormal;
 local vector HitLocation;
 local vector THitNormal;
 local vector THitLocation;
 local DrivableFeedback fb;
 local actor groundobject,furtherground,hitobject;
 local vector g;
 local float traction;
 local rotator AdjustRotation;
 local rotator actualrotation;
 local rotator yawrelativerotation;
 local rotator temprot;
 local float temproll;
 local vector temploc;
 local vector N,O;
 local vector TempLocation,upvect;
 local bool hovering;

 groundobject=None;
 furtherground=None;
 
 PreviousLocation = CurrentLocation;

 AdJustRotation=rot(0,0,0);
 // when in the water, self righting

 if(region.zone!=None) {
  if(region.zone.bWaterZone)
   traction=watertraction;
  else
   traction=airtraction;
 }
 else
  traction=1;

 hovering=false;
 
 if(PhysicsHover && Feedback[groundfeedbackindex].act!=None) {
  if(Feedback[groundfeedbackindex].act.region.zone.bWaterZone) {
   hovering=true;
   traction=watertraction;
  }
  else
   traction=airtraction;
 }
 
 if(Feedback[groundfeedbackindex].act!=None) {
  if(DrivableFeedback(Feedback[groundfeedbackindex].act).bIsGroundFeedback) {

   CurrentLocation = CurrentLocation+deltatime*(linearvelocity+gravityinducedvelocity); 
   TempLocation    = Feedback[groundfeedbackindex].originoffset;
   upvect=TempLocation;
   upvect.Z=0;
   N=PreviousLocation+RealWorldLocation(RealWorldLocation(TempLocation+vect(0,0,2.5),PreviousRotation),planeangles);
//   N=PreviousLocation+RealWorldLocation(RealWorldLocation(upvect,PreviousRotation),planeangles);
   O=PreviousLocation+RealWorldLocation(RealWorldLocation(TempLocation-vect(0,0,2.5),PreviousRotation),planeangles);
   if(passengerof!=None) {
    N=RealWorldLocation(N,passengerof.rotation)+passengerof.location;
    O=RealWorldLocation(O,passengerof.rotation)+passengerof.location;
   }

   groundobject=Trace(HitLocation,HitNormal,O,N,false);
   while(groundobject==Self) {
    groundobject=Trace(HitLocation,HitNormal,O,HitLocation+normal(O-HitLocation),false);
   }
   if(region.zone.bWaterZone || hovering || groundobject!=None && groundobject!=self && rotator(HitNormal).Pitch>8192) { // i.e. currently on the ground
    if(region.zone.bWaterZone && !hovering) 
     gravityinducedvelocity=-normal(region.zone.zonegravity);//*buoyancy/100;
    else
    if(groundobject!=None && Groundobject!=Self && rotator(HitNormal).Pitch>8192)
     gravityinducedvelocity=vect(0,0,0);
    else
     gravityinducedvelocity=normal(region.zone.zonegravity);
    traction=1;
    temploc = VirtualLocation(HitNormal,currentrotation)+vect(0,0,1);
    AdjustRotation= rot(0,0,0);
    AdjustRotation.roll  = normalizeangle(16384.0-atan2(temploc.Z,temploc.Y)/pi*32768.0);
    AdjustRotation.pitch = normalizeangle(atan2(sqrt(temploc.Z*temploc.Z + temploc.Y*temploc.Y),temploc.X)/pi*32768.0-16384.0);
    passedgroundtime=0;
   }
   else {
    if(passedgroundtime>0.25) {
     if(region.zone.bWaterZone) 
      gravityinducedvelocity=-normal(region.zone.zonegravity)*buoyancy/100.0;
     else
      gravityinducedvelocity+=region.zone.zonegravity*deltatime;
    }
    else {
     if(region.zone.bWaterZone) 
      gravityinducedvelocity=-normal(region.zone.zonegravity)*buoyancy/100.0;
    
     passedgroundtime+=deltatime;
    }
   }
   MovementDirection=rotator(CurrentLocation-PreviousLocation);
   N=PreviousLocation+RealWorldLocation(RealWorldLocation(TempLocation+vect(0,0,2.5),PreviousRotation),planeangles);
   O=CurrentLocation +RealWorldLocation(RealWorldLocation(TempLocation+vect(0,0,2.5),CurrentRotation),planeangles);
   if(passengerof!=None) {
    N=RealWorldLocation(N,passengerof.rotation)+passengerof.location;
    O=RealWorldLocation(O,passengerof.rotation)+passengerof.location;
   }

   fb = DrivableFeedback(Feedback[groundfeedbackindex].Act);
   
   groundobject=Trace(HitLocation,HitNormal,O,N,false);
   while(groundobject==Self) {
    groundobject=Trace(HitLocation,HitNormal,O,HitLocation+normal(O-HitLocation),false);
   }
      
   if(groundobject!=None && groundobject!=self && rotator(HitNormal).Pitch>8192) { // on the ground
    temploc = VirtualLocation(HitNormal,currentrotation)+vect(0,0,1);
    AdjustRotation= rot(0,0,0);
    AdjustRotation.roll  = normalizeangle(16384.0-atan2(temploc.Z,temploc.Y)/pi*32768.0);
    AdjustRotation.pitch = normalizeangle(atan2(sqrt(temploc.Z*temploc.Z + temploc.Y*temploc.Y),temploc.X)/pi*32768.0-16384.0);
    CurrentLocation=HitLocation-RealWorldLocation(RealWorldLocation(Feedback[groundfeedbackindex].originoffset,CurrentRotation),planeangles);
    if(passengerof!=None)
     CurrentLocation-=passengerof.location;
    traction=1;
    passedgroundtime=0;
   }
   else {
    if(groundobject!=None) { // hit vertical obstacle
     N=CurrentLocation;
     O=CurrentLocation+RealWorldLocation(RealWorldLocation(TempLocation+vect(0,0,2.5),PreviousRotation),planeangles);
     groundobject=Trace(HitLocation,HitNormal,O,N,false);
     while(groundobject==Self) {
      groundobject=Trace(HitLocation,HitNormal,O,HitLocation+normal(O-HitLocation),false);
     }
     if(groundobject!=None) {
      CurrentLocation=HitLocation-RealWorldLocation(RealWorldLocation(Feedback[GroundFeedbackIndex].originoffset,CurrentRotation),PlaneAngles);
      if(AdjustRotation!=rot(0,0,0))
       CurrentLocation+=RealWorldLocation(RealWorldLocation(vect(0,0,1),CurrentRotation+adjustrotation),planeangles);;
     }
//        CurrentLocation=HitLocation-RealWorldLocation(RealWorldLocation(Feedback[groundfeedbackindex].originoffset,CurrentRotation),planeangles);
    }
    groundobject=None;
   }
   if(groundobject==self)
    groundobject=None;

    

  }
  if(groundobject==None && (region.zone.bWaterZone || Hovering)) {
   AdjustRotation.pitch = -CurrentRotation.Pitch;
   AdjustRotation.roll = -CurrentRotation.roll;
   if(AdjustRotation.Pitch<0) 
    AdjustRotation.Pitch=MAX(AdjustRotation.Pitch,-PhysicalRotationRate.Pitch*deltatime);
   else
   if(AdjustRotation.Pitch>0) 
    AdjustRotation.Pitch=MIN(AdjustRotation.Pitch,PhysicalRotationRate.Pitch*deltatime);
  
   if(AdjustRotation.Roll<0) 
    AdjustRotation.Roll=MAX(AdjustRotation.Roll,-PhysicalRotationRate.Roll*deltatime);
   else
   if(AdjustRotation.Roll>0) 
    AdjustRotation.Roll=MIN(AdjustRotation.Roll,PhysicalRotationRate.Roll*deltatime);
  }
  CurrentRotation = CurrentRotation+AdjustRotation;
  currenttraction=traction;

  if(traction>0) {
   if(!region.zone.bWaterZone)
    UpdateSpeed(deltatime,1);
   else
    UpdateSpeed(deltatime,traction);
    
  }
   
  if(CheckForCollision(deltatime)) {
   CurrentLocation = PreviousLocation;
   gravityinducedvelocity=vect(0,0,0);
   if(groundobject!=None) {                 // don't pitch/roll when on the ground
    CurrentRotation = PreviousRotation;
   }
   else {                                   // assume the world wants to rotate us 
    AdjustRotation.pitch = -CurrentRotation.Pitch; 
    AdjustRotation.roll = -CurrentRotation.roll;
    if(AdjustRotation.Pitch<0) 
     AdjustRotation.Pitch=MAX(AdjustRotation.Pitch,-PhysicalRotationRate.Pitch*deltatime);
    else
    if(AdjustRotation.Pitch>0) 
     AdjustRotation.Pitch=MIN(AdjustRotation.Pitch,PhysicalRotationRate.Pitch*deltatime);
  
    if(AdjustRotation.Roll<0) 
     AdjustRotation.Roll=MAX(AdjustRotation.Roll,-PhysicalRotationRate.Roll*deltatime);
    else
    if(AdjustRotation.Roll>0) 
     AdjustRotation.Roll=MIN(AdjustRotation.Roll,PhysicalRotationRate.Roll*deltatime);
//    Also bounce up
//    gravityinducedvelocity=-region.zone.zonegravity*deltatime;    
   }
   linearvelocity = vect(0,0,0);
   CurrentSpeed=0;
  }

  if(vsize(linearvelocity)<2)
   linearvelocity=vect(0,0,0);
   
  velocity = linearvelocity+gravityinducedvelocity;
 }
}


// Looks for an edge by drawing a line from StartPoint to EndPoint and looking for a hitlocation
simulated final function bool FindEdge(vector StartPoint,vector EndPoint,out vector Edge)
{
 local vector HitLocation,HitNormal;
 local actor act;
 
 act=Trace(HitLocation,HitNormal,EndPoint,StartPoint,false);
 if(act!=None) {
  Edge=HitLocation;
  return true;
 }
 return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// TerrainFollowing physics model.
// Can be used for cars, tanks. (not for motorcycles)
// 5 special feedback points are needed, in addition to the collision points:
// One groundfeedback, 4 auxiliary groundfeedbacks, all on different sides of the groundfeedback.
// The pattern does not need to be rectangular, or symmetrical, as long as any three points
// combined result in a stable base.
// You may use more than 4 feedback points, but feedback points must be available in all four quadrants
// There is special handling of the ground feedback in case PhysicsHover is enabled (hovercraft hovers on water)
///////////////////////////////////
simulated function TFPhysics(float DeltaTime)
{
 local float accel;
 local vector LinearAcceleration,NewLocation;
 local rotator AngularAcceleration;
 local float collisiontime;
 local rotator MovementDirection;
 local vector HitNormal;
 local vector HitLocation;
 local vector THitNormal;
 local vector THitLocation;
 local DrivableFeedback fb;
 local actor groundobject,furtherground,hitobject;
 local vector g;
 local float traction;
 local rotator AdjustRotation;
 local rotator actualrotation;
 local rotator yawrelativerotation;
 local rotator temprot;
 local float temproll;
 local vector temploc;
 local vector N,O,FeedbackOffset,REF;
 local vector TempLocation,upvect;
 
 local int i;
 local bool CurrentlyOnGround,CurrentlyStable,GroundFBonground,hovering;        // any criteria
 local vector VirtualGroundLocation;  // superimposed 
 local int AuxiliaryGroundCount;      // how many auxiliary feedbacks are on the ground
 local vector VirtualVelocity;
 local vector NewFeedbackLocation,ownlocation;
 local rotator ownrotation;
 local int FirstFeedback;
 local float BackZ,FrontZ,LeftZ,RightZ,BackDist,FrontDist,LeftDist,RightDist,tempdist,groundZ;
 local int BackFB,FrontFB,LeftFB,RightFB;
 local int BackCount,FrontCount,LeftCount,RightCount;
 local float tempang,feedbackdist;
 local float frontang,backang,leftang,rightang;
 local float gravdist;
 
 groundobject=None;
 furtherground=None;
 
 PreviousLocation = CurrentLocation;

 AdjustRotation=rot(0,0,0);
 CurrentlyOnGround=false;
 CurrentlyStable=false;
 hovering=false;

 if(region.zone!=None) {
  if(region.zone.bWaterZone)
   traction=watertraction;
  else
   traction=airtraction;
 }
 else
  traction=1;

 CurrentLocation+=(linearvelocity+gravityinducedvelocity)*deltatime;

 ownrotation=CalculateOwnRotation(); 
 ownlocation=CalculateOwnLocation(); 

 gravdist=VirtualLocation(region.zone.zonegravity,ownrotation).Z; // no logic here, this just seems to work well
//SetCollision(false,false,false);

 ///////////////////////////////////////////////////////////////////////////
 // Now check the new location by dropping down some lines
 GroundZ=0; BackZ=0;  FrontZ=0; LeftZ=0; RightZ=0;
 FrontCount=0; LeftCount=0; BackCount=0; RightCount=0;
 RightDist=0; FrontDist=0; BackDist=0; LeftDist=0;
 leftang=0; rightang=0; frontang=0; backang=0;
 firstfeedback=-1;
 GroundFBonground=false;
 adjustrotation=rot(0,0,0);
 if(groundfeedbackindex>=0 && PhysicsFlatBottom) {
  TempLocation    = Feedback[i].originoffset;
  REF=RealWorldLocation(RealWorldLocation(Feedback[groundfeedbackindex].originoffset,CurrentRotation)+CurrentLocation,planeangles);
  N=CurrentLocation;
  TempLocation    = Feedback[groundfeedbackindex].originoffset;
  Feedback[groundfeedbackindex].originoffset.Z-=2.0;   // =0
  N=CalculateChildLocation(Feedback[groundfeedbackindex],OwnLocation,OwnRotation);
  Feedback[groundfeedbackindex].originoffset.Z=TempLocation.Z+gravdist; //2; // *2;
  O=CalculateChildLocation(Feedback[groundfeedbackindex],OwnLocation,OwnRotation);
  Feedback[groundfeedbackindex].originoffset.Z=TempLocation.Z;
  REF=CalculateChildLocation(Feedback[groundfeedbackindex],OwnLocation,OwnRotation);
  groundobject=Trace(HitLocation,HitNormal,O,N,false);
  while(drivable(groundobject)!=None && drivable(groundobject).MainDrivable==Self && VSize(O-HitLocation)>1.5) {
   N=HitLocation+Normal(O-HitLocation);
   groundobject=Trace(HitLocation,HitNormal,O,N,false);
  }
  if(GroundObject!=None && (drivable(groundobject)==None || drivable(groundobject).MainDrivable!=Self)) {
   GroundZ=VirtualLocation(HitLocation-REF,ownrotation).Z;
   GroundFBOnGround=true;
  }
 }
 if(groundfeedbackindex>=0 && PhysicsHover) {
  if(Feedback[groundfeedbackindex].act.region.zone.bWaterZone) {
   GroundZ=Feedback[groundfeedbackindex].originoffset.Z;
   GroundFBOnGround=true;
   Hovering=true;
  }
 }
 for(i=0;i<feedbackcount;i++) {
  if(drivablefeedback(feedback[i].act).bIsAuxiliaryFeedback) {
   TempLocation    = Feedback[i].originoffset;
   Feedback[i].originoffset.Z=0;
   N=CalculateChildLocation(Feedback[i],OwnLocation,OwnRotation);

   if(Hovering)
    Feedback[i].originoffset.Z=TempLocation.Z;
   else
    Feedback[i].originoffset.Z=TempLocation.Z+gravdist; // *2;
    
   O=CalculateChildLocation(Feedback[i],OwnLocation,OwnRotation);
   Feedback[i].originoffset.Z=TempLocation.Z;
   REF=CalculateChildLocation(Feedback[i],OwnLocation,OwnRotation);
   Feedback[i].act.SetLocation(REF);

   groundobject=Trace(HitLocation,HitNormal,O,N,false);
   while(drivable(groundobject)!=None && drivable(groundobject).MainDrivable==Self && VSize(O-HitLocation)>1.5) {
    N=HitLocation+Normal(O-HitLocation);
    groundobject=Trace(HitLocation,HitNormal,O,N,false);
   }
   if(GroundObject!=None && (drivable(groundobject)==None || drivable(groundobject).MainDrivable!=Self)) {
    tempdist=VirtualLocation(HitLocation-REF,OwnRotation).Z; 
    if(FirstFeedback==-1 || FeedbackDist<tempdist) { // note that a positive tempdist pushes up, and a negative one pulls down
     FirstFeedback=i;
     FeedbackDist=tempdist;
    }
    if(Feedback[i].originoffset.X>0) {
     if(tempdist>frontZ || frontCount==0) {
      FrontZ=tempdist;
      FrontDist=Feedback[i].originoffset.X;
      FrontFB=i;
     }
     FrontCount=1;
    }
    if(Feedback[i].originoffset.X<0) {
     if(tempdist>backZ || backCount==0) {
      BackZ=tempdist;
      BackDist=Feedback[i].originoffset.X;
      BackFB=i;
     }
     BackCount=1;
    }
    if(Feedback[i].originoffset.Y>0) {
     if(tempdist>rightZ || RightCount==0) {
      RightZ=tempdist;
      RightDist=Feedback[i].originoffset.Y;
      RightFB=i;
     }
     RightCount=1;
    }
    if(Feedback[i].originoffset.Y<0) {
     if(tempdist>leftZ || LeftCount==0) {
      LeftZ=tempdist;
      LeftDist=Feedback[i].originoffset.Y;
      LeftFB=i;
     }
     LeftCount=1;
    }
   }
  }
 }

 if(FirstFeedback>=0) {
  if(GroundFBOnGround) {
   if(BackCount==0 && FrontCount>0) {
    BackZ=GroundZ;
    BackCount=1;
    BackDist=0;
    BackFB=groundfeedbackindex;
   }
   if(FrontCount==0) {
    FrontZ=GroundZ;
    FrontCount=1;
    FrontDist=0;
    FrontFB=groundfeedbackindex;
   }
   if(LeftCount==0 && RightCount>0) {
    LeftZ=GroundZ;
    LeftCount=1;
    LeftDist=0;
    LeftFB=groundfeedbackindex;
   }
   if(RightCount==0) {
    RightZ=GroundZ;
    RightCount=1;
    RightDist=0;
    RightFB=groundfeedbackindex;
   }
  }    
  // actual internal angle between the feedbacks in question
  tempang=32768*atan2(Feedback[FrontFB].originoffset.Z-Feedback[BackFB].originoffset.Z,Feedback[FrontFB].originoffset.X-Feedback[BackFB].originoffset.X)/pi;
//  tempang=0;   

  if(BackCount>0 && FrontCount>0) // stable
   adjustrotation.Pitch=normalizeangle(32768.0*atan2(FrontZ-BackZ,FrontDist-BackDist)/pi-tempang);
  else 
  if(FrontCount>0)
   adjustrotation.Pitch=normalizeangle(32768.0*atan2(square(deltatime)*VSize(region.zone.zonegravity)/2.0+deltatime*vsize(gravityinducedvelocity),FrontDist-BackDist)/pi);
  else
  if(BackCount>0)
   adjustrotation.Pitch=-normalizeangle(32768.0*atan2(square(deltatime)*VSize(region.zone.zonegravity)/2.0+deltatime*vsize(gravityinducedvelocity),FrontDist-BackDist)/pi);

  if(FrontCount>0) {
   adjustrotation.pitch+=CurrentSpeed; //
  }
  if(BackCount>0) {
   adjustrotation.pitch-=CurrentSpeed; //
  }  
  tempang=32768*atan2(Feedback[RightFB].originoffset.Z-Feedback[LeftFB].originoffset.Z,Feedback[RightFB].originoffset.Y-Feedback[LeftFB].originoffset.Y)/pi;
//  tempang=0;

  if(LeftCount>0 && RightCount>0) // stable
   adjustrotation.Roll=-normalizeangle(32768.0*atan2(RightZ-LeftZ,RightDist-LeftDist)/pi-tempang); 
  else 
  if(RightCount>0)
   adjustrotation.Roll=-normalizeangle(32768.0*atan2(square(deltatime)*VSize(region.zone.zonegravity)/2.0+deltatime*vsize(gravityinducedvelocity),RightDist-LeftDist)/pi);
  else 
  if(LeftCount>0)
   adjustrotation.Roll=normalizeangle(32768.0*atan2(square(deltatime)*VSize(region.zone.zonegravity)/2.0+deltatime*vsize(gravityinducedvelocity),RightDist-LeftDist)/pi);
 
  if(BackCount>0 && FrontCount>0) {
   if(RightCount>0 && LeftCount==0)
    CurrentRotation.yaw+=4.0*CurrentSpeed;
   else
   if(LeftCount>0 && RightCount==0) 
    CurrentRotation.yaw-=4.0*CurrentSpeed;
  }
   
  adjustrotation.Pitch+=(DesiredSpeed-CurrentSpeed); // this makes the front rise a bit when accelerating

  if(RightCount>0 && LeftCount>0 || FrontCount>0 && BackCount>0 || GroundFBOnGround) {
   CurrentlyOnGround=true;
  }
 }
 else { 
  if(region.zone.bWaterZone || Hovering && FirstFeedback==-1) {
   AdjustRotation.pitch = -CurrentRotation.Pitch;
   AdjustRotation.roll = -CurrentRotation.roll;
  }
 }
 if(GroundFBOnGround && FirstFeedback==-1)
  FirstFeedback=groundfeedbackindex;
  
////////////////////// assertion
// if(CurrentController!=None) {
//  CurrentController.ReportedProjectileLoaded=(FrontCount>0); //CurrentlyOnGround;
//  CurrentController.ReportedAltProjectileLoaded=(BackCount>0); //GroundFBOnGround;
// }

// cap the rotation rate
 if(AdjustRotation.Pitch<0) 
  AdjustRotation.Pitch=MAX(AdjustRotation.Pitch,-PhysicalRotationRate.Pitch*deltatime);
 else
 if(AdjustRotation.Pitch>0) 
  AdjustRotation.Pitch=MIN(AdjustRotation.Pitch,PhysicalRotationRate.Pitch*deltatime);
 
 if(AdjustRotation.Roll<0) 
  AdjustRotation.Roll=MAX(AdjustRotation.Roll,-PhysicalRotationRate.Roll*deltatime);
 else
 if(AdjustRotation.Roll>0) 
  AdjustRotation.Roll=MIN(AdjustRotation.Roll,PhysicalRotationRate.Roll*deltatime);

 CurrentRotation = CurrentRotation+AdjustRotation;
 ownrotation=CalculateOwnRotation(); 
 ownlocation=CalculateOwnLocation(); 
 gravdist=VirtualLocation(region.zone.zonegravity,ownrotation).Z; // no logic here, this just seems to work well
 if(CurrentlyOnGround) {
  i=FirstFeedback;
  LeftCount=0;
  for(i=0;i<feedbackcount;i++) {
   if(drivablefeedback(feedback[i].act).bIsAuxiliaryFeedback) {
    REF=CalculateChildLocation(Feedback[i],CurrentLocation,CurrentRotation);
// Now drop a line from one of the auxiliary feedbacks                 
    TempLocation    = Feedback[i].originoffset;
    Feedback[i].originoffset.Z=0;
    N=CalculateChildLocation(Feedback[i],OwnLocation,OwnRotation);

    if(hovering)
     Feedback[i].originoffset.Z=TempLocation.Z;//*2;
    else
     Feedback[i].originoffset.Z=TempLocation.Z+gravdist;//*2;

    O=CalculateChildLocation(Feedback[i],OwnLocation,OwnRotation);
    Feedback[i].originoffset.Z=TempLocation.Z;
    REF=CalculateChildLocation(Feedback[i],OwnLocation,OwnRotation);
//  O+=square(deltatime)*region.zone.zonegravity/2+deltatime*gravityinducedvelocity;
    groundobject=Trace(HitLocation,HitNormal,O,N,false);
    while(drivable(groundobject)!=None && drivable(groundobject).MainDrivable==Self && VSize(O-HitLocation)>1.5) {
     N=HitLocation+Normal(O-HitLocation);
     groundobject=Trace(HitLocation,HitNormal,O,N,false);
    }
    if(GroundObject!=None && (drivable(groundobject)==None || drivable(groundobject).MainDrivable!=Self)) {
     if(upvect.Z<(HitLocation-REF).Z || LeftCount==0) {
      upvect=HitLocation-REF;
      LeftCount=1;
     }
    }
   }
  }
  if(LeftCount==0)
   CurrentlyOnGround=false;
 }
 if(VSize(region.zone.zonegravity)>0 && CurrentlyOnGround)
  traction=max(0,-VirtualLocation(region.zone.zonegravity,ownrotation).Z)/VSize(region.zone.zonegravity);   // this means 
 else {
  if(region.zone.bWaterZone)
   traction=watertraction;
  else
   traction=airtraction;
 }
  
 if(CurrentlyOnGround) { // a few statements back, currentlyOnGround is reset if no feedback hits the floor after rotation 
  CurrentLocation += VirtualLocation(upvect,ownrotation);
//   TempLocation    = Feedback[FirstFeedback].originoffset;
//   CurrentLocation=HitLocation-RealWorldLocation(RealWorldLocation(TempLocation,CurrentRotation),planeangles);
    
  CurrentLocation+=RealWorldLocation(vect(0,0,1),CurrentRotation);
  gravityinducedvelocity=vect(0,0,0);
  passedgroundtime=0;
 }
 else {
  if(Hovering)  
   gravityinducedvelocity=-normal(feedback[groundfeedbackindex].act.region.zone.zonegravity)*buoyancy/100.0;
  else
  if(region.zone.bWaterZone) 
   gravityinducedvelocity=-normal(region.zone.zonegravity)*buoyancy/100.0;
  else
   GravityInducedVelocity+=region.zone.zonegravity*deltatime;
 }
 currenttraction=traction;

  

 if(traction>0) {
//  if(!region.zone.bWaterZone)
//   UpdateSpeed(deltatime,1);
//  else
   UpdateSpeed(deltatime,traction);
   
 } // traction>0
   
 if(CheckForCollision(deltatime)) { // ||
//  abs(normalizeangle(adjustrotation.pitch+currentrotation.pitch))>8192 ||
//  abs(normalizeangle(adjustrotation.roll+currentrotation.roll))>8192) {
  gravityinducedvelocity=vect(0,0,0);
  CurrentRotation = LastGoodRotation;
  CurrentLocation = LastGoodLocation;
  linearvelocity = -linearvelocity; // vect(0,0,0);
  CurrentSpeed=-CurrentSpeed;
 } // checkforcollision
 else {
// if(CurrentlyOnGround) {
//  LastGoodLocation=PreviousLocation;
//  LastGoodRotation=PreviousRotation;
  LastGoodLocation=CurrentLocation;
  LastGoodRotation=CurrentRotation;
 }

 if(vsize(linearvelocity)<2)
  linearvelocity=vect(0,0,0);
   
//SetCollision(true,true,true);
 velocity = linearvelocity+gravityinducedvelocity;
}



// ForceCalculation physics model: we have a speed, we're travelling through the world until we hit something.
// at that point, we calculate the forces acting upon us, which results in an updated speed and rotational speed.
// The rest is left to the next DoPhysics call, i.e. we automatically wait ("shake") when we hit something.
// The alternative would be to calculate the forces for every 'key-point' in time, but this may slow the game
// down too much. (don't know yet)
// 'ForceCalculation' method is pretty advanced: it allows for propulsion in one direction, lift, gravity,
// and will follow the terrain by 'bouncing'
simulated function ForceCalculationPhysics(float DeltaTime)
{
 local float accel;
 local vector LinearAcceleration,NewLocation;
 local rotator AngularAcceleration;
 local float collisiontime;
 local rotator MovementDirection;
 
 if(DeltaTime<=0)
  return;
   
 if(!bPhysics)
  return;

 if(parentdrv!=None) 
  return;

 PreviousLocation = CurrentLocation;

 MovementDirection=rotator(linearvelocity);
 // calculate speed
 CurrentSpeed = virtuallocation(linearvelocity,rotation).X;

 // default method automatically 'applies the brakes' to slow down 
 accel=0.0;
 if(MaxSpeed>0) {
  if(DesiredSpeed<CurrentSpeed)
   accel=MAX(DesiredSpeed-CurrentSpeed,-MaxDeceleration*deltatime); // note: desiredspeed-currentspeed is negative, so accel is negative

  if(DesiredSpeed>CurrentSpeed)
   accel=MIN(DesiredSpeed-CurrentSpeed,MaxAcceleration*deltatime); 
 }

 CurrentLocation = PreviousLocation+deltatime*linearvelocity + deltatime*deltatime*(vector(rotation)*accel/*+region.zone.zonegravity*/)/2.0; // predict what acceleration and gravity will do
 linearvelocity += vector(Rotation)*accel+deltatime*region.zone.zonegravity;
 CheckForFeedBack(deltatime,AngularAcceleration,LinearAcceleration,NewLocation,collisiontime);
 CurrentLocation = NewLocation;
//  CurrentLocation = PreviousLocation;

 linearvelocity += LinearAcceleration;    // note that LinearAcceleration and AngularAcceleration already have  
 angularvelocity += AngularAcceleration;  // been multiplied by deltatime/collisiontime

 if(vsize(linearvelocity)<2)
  linearvelocity=vect(0,0,0);
  
 velocity = linearvelocity;
}

// Collision detection and force calculation. Not using standard Unreal collision detection, since
// most vehicles don't resemble cilinders.
final function bool CheckForFeedBack(float deltatime,out rotator TotalAngularAcceleration,out vector TotalLinearAcceleration,out vector NewLocation,out float bouncetime)
{
 local int i;
 local float keypoint; // keypoint is moment of collision with actor/world
 local actor act_t;
 local actor earliestactor;
 local float accel;
 local vector O;
 local vector HitNormal;
 local vector HitLocation;
 local DrivableFeedback fb;
 local rotator MovementDirection;
 local float earliestcollision;
 local vector TempSpeed;
 local vector LinearAcceleration;
 local rotator AngularAcceleration;
 local int earliestcount;
 local vector pivotvect[50];
 local rotator pivotrot[50];
 local int pivotcount;
 local rotator temprot;
 local vector temploc;
 local int j;
 local vector pivotpoint;
 local bool found;
 local vector TempAcceleration;
 local vector curloc,prevloc;
 local rotator currot,prevrot;
 
 earliestcollision=1000000;
 earliestactor=None;
 earliestcount=0;
 
 MovementDirection = rotator(linearvelocity);
 prevloc=RealWorldLocation(PreviousLocation,previousplaneangles)+previousOrigin;
 prevrot=RealWorldRotation(PreviousRotation,previousplaneangles);
 curloc= RealWorldLocation(CurrentLocation,planeangles)+Origin;
 currot= RealWorldRotation(CurrentRotation,planeangles);

 for(i=0;i<FeedbackCount;i++) { // check all of our DrivableFeedback points for collisions
  O=CalculateChildLocation(Feedback[i],prevloc,prevrot);
//  N=CalculateChildLocation(Feedback[i],curloc,currot);
//  O=Feedback[i].originoffset;             // where is the child's origin in our own space
//  O=RealWorldLocation(O,PreviousRotation);// We may have rotated in our own plane
//  O=RealWorldLocation(O,planeangles);    // Our own plane may have been rotated
//  O+=RealWorldLocation(PreviousLocation,planeangles)+Origin;
  fb = DrivableFeedback(Feedback[i].Act);
  if(!fb.bIsGroundFeedback) {
   // next test determines if feedbackpoint points roughly in the same direction as the movement is
   if(VirtualLocation(vector(fb.Rotation),MovementDirection).X<0) {
    act_t=Trace(HitLocation,HitNormal,O,fb.Location, false);
    fb.hitactor = act_t;
    if(act_t!=None) { //collision
     if(act_t==earliestactor)
      earliestcount++;
     keypoint = deltatime*vsize(HitLocation-fb.Location)/vsize(O-fb.Location);
     if(keypoint<=earliestcollision || earliestactor == None) {
      if(earliestactor!=act_t)
       earliestcount=1;
      earliestcollision=keypoint;
      fb.hittime = keypoint;
      earliestactor = act_t;
      fb.hitactor = act_t;
      fb.hitlocation = hitlocation;
      fb.hitnormal = hitnormal;
     }
    }
   }
   else {
    fb.hitactor = None;
    fb.hittime=-100000;  // unlikely that this deltatime ever occurs...
   }
  }
 } // collision detection loop
 
 pivotpoint=vect(0,0,0);
 pivotcount=0;
 if(earliestactor!=None) {
  for(i=0;i<FeedbackCount;i++) {
   fb= DrivableFeedback(Feedback[i].act);
   if(!fb.bIsGroundFeedback) {
    if(fb.hitactor == earliestactor) { 
     temprot=virtualrotation(feedback[i].originangles,movementdirection);
     temploc=virtuallocation(feedback[i].originoffset,movementdirection);
     found=false;
     for(j=0;j<pivotcount;j++) {
      if(pivotrot[j].Yaw==temprot.Yaw) {
       if(vsize(pivotvect[j])>vsize(temploc)) { // if temploc is closer to the center, pivottemp will not have any effect
        pivotvect[j]=temploc;
        pivotrot[j]=temprot;
       }
       found=true;
      }
     }
     if(!found && pivotcount<arraycount(pivotvect)) {
      pivotvect[pivotcount]=temploc;
      pivotrot[pivotcount]=temprot;
      pivotcount++;
     }
    }
   }
  }
  for(i=0;i<pivotcount;i++)
   pivotpoint+=pivotvect[i];
  if(pivotcount>1) // we don't have to calculate an average if there are less than 2 feedback points
   pivotpoint = pivotpoint/pivotcount;
 }
////////////////////////////////////////////////////////////////////////////////////////
// calculate effect of world 
 bouncetime = MIN(0.1,earliestcollision-deltatime);
 TempSpeed = virtuallocation(linearvelocity,rotator(pivotpoint));
 TempSpeed.Y=0;
 TempSpeed.Z=0;
 TempSpeed.X = -(1+Elasticity)*TempSpeed.X;
 TempAcceleration=realworldlocation(TempSpeed,rotator(pivotpoint));
 SplitAcceleration(pivotpoint,TempAcceleration,LinearAcceleration,AngularAcceleration);
 TotalLinearAcceleration=LinearAcceleration;
 TotalAngularAcceleration=AngularAcceleration;

 // Look at all feedback points for lift
 for(i=0;i<FeedbackCount;i++) {
  fb= DrivableFeedback(Feedback[i].act);
  if(!fb.bIsGroundFeedback) {
   SplitAcceleration(feedback[i].originoffset,fb.lift(),LinearAcceleration,AngularAcceleration);
   TotalLinearAcceleration+=LinearAcceleration;
   TotalAngularAcceleration+=AngularAcceleration; 
  }
 } // for  
 NewLocation = CurrentLocation;

 if(earliestactor!=None) {
  NewLocation = PreviousLocation;
  //NewLocation += TotalLinearAcceleration*1.01;// extra 1% is used to push clear from the wall
  return true;
 }
 bouncetime=0;
 return false;
}



///////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////// Aiming functions ///////////////////////////////////////
simulated function BallisticCalculation(float latitude,float longitude,float altitude,out float yaw,out float hightraj,out float flighttime)
{
 local vector tempvect;
 
// log(latitude $ " " $ longitude $ " " $ altitude);
 tempvect.X=latitude-location.x;
 tempvect.Y=longitude-location.y;
 tempvect.Z=altitude-location.z;

 HighBallisticSolution(sqrt(square(tempvect.X)+square(tempvect.Y)),tempvect.Z,hightraj,flighttime);
 Yaw = 180.0*atan2(tempvect.Y,tempvect.X)/pi; 
 HighTraj=HighTraj*180.0/32768.0;
}

// uses iterations to find the angles under which a projectile needs to be fired
simulated final function HighBallisticSolution(float l,float h,out float high_angle,out float flighttime)
{
 local float g;
 local float v;
 local int step;
 local float angle;
 local float h_n,drop,corr;
 local float L1,L2,stepsize;
 local float min_angle,max_angle,mindist,maxdist;
 
 
 g=vsize(region.zone.zonegravity);
 v=ProjectileSpeed;
 
 high_angle=16384;

 if(v==0) // i.e if there is no way to get anywhere
  return;
  
 if(g==0) {// i.e. if there is no gravity
  high_angle = 16384.0*atan2(h,l)/pi; // instant hit angle
  return;
 }

 if(l==0) { // special case: straight up
  high_angle=16384.0;
  return;
 }

 if(h>v*v/g) // i.e. if height is greater than maximum height
  return;
  
 step=0;
 min_angle=pi/4.0;
 max_angle=pi/2.0;
 mindist=0;
 predict_distance_travelled(h,g,v,angle,L1,maxdist);
 do {
  step++;
  angle=(min_angle+max_angle)/2.0;
  predict_distance_travelled(h,g,v,angle,L1,L2);
  if(L2>L) {
   min_angle=angle;
   maxdist=L2;
  }
  else {
   max_angle=angle;
   mindist=L2;
  }
 } until(step>8); // significantly small interval for linear interpolation
 angle = ((max_angle-min_angle)*(L-mindist)/(maxdist-mindist) + min_angle);
 predict_distance_travelled(h,g,v,angle,L1,L2);
 flighttime = L2/v/cos(angle);
 high_angle = angle*32768.0/pi;
}

simulated final function DirectFireBallisticSolution(float l,float h,out float low_angle)
{
 local float g;
 local float v;
 local int step;
 local float angle;
 local float h_n,drop,corr;
 local float L1,L2,stepsize;
 local float min_angle,max_angle,mindist,maxdist;
 
 
 g=vsize(region.zone.zonegravity);
 v=ProjectileSpeed;
 
 low_angle=-32768;       // not a sensible value. test for <16384 to know if there is a solution
 
 if(v==0) // i.e if there is no way to get anywhere
  return;
  
 if(g==0) {// i.e. if there is no gravity
  low_angle = 16384.0*atan2(h,l)/pi; // instant hit angle
  return;
 }

 if(l==0) { // special case: straight up
  low_angle=16384.0;
  return;
 }

 if(h>v*v/g) // i.e. if height is greater than maximum height
  return;
  
 step=0;
 corr=0;
 angle = atan2(h+corr,l);
 do {
  step++;
  drop=g*square(l)/(2.0*square(v)*square(cos(angle)));
  h_n=h+corr-drop;
  corr+=h-h_n;
  angle = atan2(h+corr,l);
 } until(step>5 || abs(h-h_n)<1); // increase no. steps for greater accuracy

 
 low_angle = angle*32768.0/pi;
}

simulated final function float pow(float base,float exponent)
{
 return exp(exponent*loge(base));
}

simulated final function predict_distance_travelled(float h,float g,float v,float a,out float l1,out float l2)
{
 local float diff,comm;
 local float sq;
 local float derdiff,dercomm;
 
 L1=0;L2=0;
 sq= v*v*sin(a)*sin(a)-2*h/g;
 
 
 if(sq<0)  // projectile will never reach that height
  return;
    
    
 diff = v/g*cos(a)*sqrt(sq);
 comm = v*v*sin(a)*cos(a)/g;
 
 L1=comm - diff;
 L2=comm + diff; 
}

defaultproperties
{
     DestructionEffect=Class'UnrealShare.BallExplosion'
     PhysicsType=FloatAndBlock
     PhysicalRotationRate=(Pitch=8192,Yaw=8192,Roll=8192)
     CrashingSound=Sound'AmbModern.Looping.gener3'
     DestroyedSound=Sound'AmbModern.Looping.Lhiss1'
     ControlFactorTurretTurn=1.000000
     ControlFactorSpeed=1.000000
     ControlFactorTurretUp=1.000000
     ControlFactorVehicleUp=1.000000
     ControlFactorVehicleTurn=1.000000
     Elasticity=0.250000
     bPhysics=True
     bRepairable=True
     RepairTime=15.000000
     MoverEncroachType=ME_IgnoreWhenEncroach
     MoverGlideType=MV_MoveByTime
     InitialState=None
     Tag=drivable
     bDirectional=True
     CollisionRadius=0.000000
     CollisionHeight=0.000000
     bProjTarget=True
}
