//=============================================================================
// Helicopter.
//=============================================================================
class Helicopter expands Drivable;
var () float AirDrag;
var () bool TailRotorControllable; // allows aim left/right to turn the helicopter during a hover
var () float TranslationalYawRate; // how quickly will the helicopter turn as a result of roll
var () float MaxRotorLift;         // how much lift will the rotor generate at maximum collective
var () float MaxTranslationalLift; // how much lift will there be when traveling at max speed
var () float ClimbRate;            // maximum climb rate the player indicates
var bool OnGround;
var float CollectiveSetting; // influenced by the jump/duck keys
var float ControlInputCollective;

// helicopter uses strafing and forward/backward controls for rotation and collective (rotor lift)
// uses look for pitching the rotor. (look left for skidding left, look down for forward movement)
// a separate gunner will be needed if you want a swiveling weapon.
simulated function InterpretControls(bool bJustFired,bool bFire,bool bJustAltFired,bool bAltFire,
                           float aVehicleSpeedControl,float aTurretTurn,float aVehicleTurn,
                           float aVehicleUp,float aTurretUp,bool bJustJumped,bool bJustDucked)
{
 if(aVehicleTurn<0)       // skidding. Note, at speed, this influences yaw
  controlinputRoll = max(MinRoll,MinRoll*(-aVehicleTurn/16384.0));
 else
  controlinputRoll = min(MaxRoll,MaxRoll*(aVehicleTurn/16384.0));

 if(TailRotorControllable) {  
  if(aTurretTurn<0)       // rotating by varying the force from the tail rotor. Note, at speed, this has very little effect
   controlinputYaw = (aTurretTurn/16384.0)*YawLeftRate;
  else
   controlinputYaw = (aTurretTurn/16384.0)*YawRightRate;
 }
 else
  controlinputYaw = 0.0;
  
 if(aVehicleSpeedControl<0)  // accelerating means pushing the nose down, not up, so inverted
  ControlInputPitch = min(MaxPitch,MaxPitch*(-aVehicleSpeedControl/16384.0));
 else
  ControlInputPitch = max(MinPitch,MinPitch*(aVehicleSpeedControl/16384.0));
 
 ControlInputCollective=max(min(aVehicleUp/16384.0,1.0),-1.0);
}

// UpdateSpeed overridden because helicopter can skid (all handled in dophysics)
simulated function UpdateSpeed(float DeltaTime,float traction)
{
}

// helicopter can have translational lift (lift through speed) as well as rotor lift.
// at low speeds, this isn't apparent, but at high speeds, the helicopter flies pretty much like
// a fixed wing aircraft.
simulated function DoPhysics(float DeltaTime)
{
 local vector acceleration;
 local vector tempvel,templocation,O,N,HitLocation,HitNormal;
 local rotator temprot;
 local actor groundobject;
 local float neededacceleration;
 
 if(groundfeedbackindex<0)
  return;

 PreviousLocation=CurrentLocation;

 neededacceleration=ControlInputCollective*ClimbRate-linearvelocity.Z;
 templocation=RealWorldLocation(vect(0,0,1),CurrentRotation);

 // calculate collective setting to counter gravity AND climb at a certain rate
 if(templocation.Z>0) {
  collectivesetting=min(1.0,max(0.0,(neededacceleration+VSize(region.zone.zonegravity))/MaxRotorLift/TempLocation.Z));
 }

 tempvel=virtuallocation(linearvelocity,rotation);
 
 if(passengerof==None)
  tempvel += virtuallocation(passengerof.velocity,rotation);

 tempvel.Z=0.0;
 
 acceleration=templocation*(collectivesetting*MaxRotorLift+VSize(tempvel)*MaxTranslationalLift/MaxSpeed)+region.zone.zonegravity;

 if(!OnGround && acceleration.Z<=0.0) { // check if we're now on the ground
  TempLocation    = Feedback[groundfeedbackindex].originoffset;
  TempLocation.X=0; 
  TempLocation.Y=0; 
  TempLocation.Z+=2.0;
  N=CurrentLocation+RealWorldLocation(RealWorldLocation(TempLocation,CurrentRotation),planeangles);
  TempLocation    = Feedback[groundfeedbackindex].originoffset;
  TempLocation.X=0; 
  TempLocation.Y=0; 
  TempLocation.Z-=2.0;
  O=CurrentLocation+RealWorldLocation(RealWorldLocation(TempLocation,CurrentRotation),planeangles);
  if(passengerof!=None) {
    N=RealWorldLocation(N,passengerof.rotation)+passengerof.location;
    O=RealWorldLocation(O,passengerof.rotation)+passengerof.location;
  }
  groundobject=Trace(HitLocation,HitNormal,O,N,false);
  while(drivable(groundobject)==Self) {
   groundobject=Trace(HitLocation,HitNormal,O,HitLocation+normal(O-HitLocation),false);
  }
  if(GroundObject!=None && groundobject!=self) 
   OnGround=true;
 }
 if(acceleration.Z>0) 
  OnGround=false;

 if(OnGround) { // on the ground, a helicopter can not move (a helicopter needs to tilt to get any movement)
  LinearVelocity=vect(0,0,0);
  super.DoPhysics(DeltaTime);
  return;
 }

// From here on, specific helicopter physics
 LastGoodLocation=CurrentLocation;
 LastGoodRotation=CurrentRotation;
 
 CurrentLocation+=(LinearVelocity)*deltatime;

 // drag
 LinearVelocity-=deltatime*normal(LinearVelocity)*MIN(AirDrag,VSize(LinearVelocity));

 // slow speed stuff
 CurrentRotation.Yaw+=ControlInputYaw*DeltaTime;

 // high speed stuff /////////////////////////////////////////
// CurrentRotation.Yaw+=VirtualLocation(acceleration,currentrotation).Y;
 tempvel=VirtualLocation(LinearVelocity,PreviousRotation);
 tempvel.Z=0;
 
 LinearVelocity+=acceleration*deltatime;

 CurrentRotation.Yaw-=CurrentRotation.Roll/MaxRoll*CurrentSpeed/MaxSpeed*TranslationalYawRate*DeltaTime;
 
 if(CurrentSpeed>MaxSpeed)
  CurrentSpeed=MaxSpeed;
 if(CurrentSpeed<-MaxSpeed)
  CurrentSpeed=-MaxSpeed;

 CurrentRotation += temprot;
 
// cap the speed
 LinearVelocity=min(VSize(LinearVelocity),MaxSpeed)*normal(LinearVelocity);
   
// Check for collision        ------- when hitting something, the helo will be damaged!
 if(CheckForCollision(deltatime)) {
  CurrentLocation = PreviousLocation;
  gravityinducedvelocity=vect(0,0,0);
  CurrentRotation = PreviousRotation;
  linearvelocity=-linearvelocity;
  if(linearvelocity.Z>0.0)
   linearvelocity.Z=0.0;
// TakeDamage(   )
 } 
}
  
simulated function RotationPhysics(float DeltaTime)
{
 local rotator NewRotation;
 local rotator NewStabilizedRotation;
 local float compensationYaw,compensationPitch,compensationRoll; // not using rotator because these values are generally<1
 local vector eyelocation,hitnormal,hitlocation,targetvector;
 local float high_angle,low_angle,flighttime;
 local rotator actualrot;
 
 if(DeltaTime<=0)
  return;
  
 if(!bPhysics)
  return;
   
 if(OnGround) // the helicopter cannot rotate while on the ground
  return;
  
 PreviousRotation = CurrentRotation;
 CurrentRotation = CurrentRotation+AngularVelocity*deltatime;
 CurrentRotation.Roll=normalizeangle(CurrentRotation.Roll);
 CurrentRotation.Pitch=normalizeangle(CurrentRotation.Pitch);
 CurrentRotation.Yaw=normalizeangle(CurrentRotation.Yaw);
 NewRotation = CurrentRotation;

 // only tail rotor is direct. Main rotor is stabilized. (meaning: if you let go of the stick, the helicopter will slow down)
 compensationyaw=ControlInputYaw*deltatime;
  
 // combined rotation rate may not exceed maximum values
 if(compensationYaw<-YawLeftRate*DeltaTime)
  compensationYaw=-YawLeftRate*DeltaTime;
 else
 if(compensationYaw>YawRightRate*DeltaTime)
  compensationYaw=YawRightRate*DeltaTime;
   
 compensationPitch=normalizeangle(ControlInputPitch-CurrentRotation.Pitch);
 compensationRoll=normalizeangle(ControlInputRoll-CurrentRotation.Roll);

 NewRotation.Roll = normalizeangle(NewRotation.Roll+compensationRoll);
 NewRotation.Pitch = normalizeangle(NewRotation.Pitch+compensationPitch);
 NewRotation.Yaw = normalizeangle(NewRotation.Yaw+compensationYaw);

 if(NewRotation.Pitch>MaxPitch+OffsetRotation.Pitch)
  NewRotation.Pitch=MaxPitch+OffsetRotation.Pitch;
 if(NewRotation.Pitch<MinPitch+OffsetRotation.Pitch)
  NewRotation.Pitch=MinPitch+OffsetRotation.Pitch;

 if(NewRotation.Roll>MaxRoll+OffsetRotation.Roll)
  NewRotation.Roll=MaxRoll+OffsetRotation.Roll;
 if(NewRotation.Roll<MinRoll+OffsetRotation.Roll)
  NewRotation.Roll=MinRoll+OffsetRotation.Roll;
  
 CurrentRotation = NewRotation; 
}

defaultproperties
{
     AirDrag=10.000000
     TranslationalYawRate=0.250000
     MaxRotorLift=1200.000000
     MaxTranslationalLift=300.000000
     ClimbRate=40.000000
     bHUD_ShowsPitch=True
     bHUD_ShowsYaw=True
     bHUD_ShowsRoll=True
     bHUD_ShowsSpeed=True
     bBot_ShowVehicleLocation=True
     Armor=50
     InitialHitPoints=80
     FullHitPoints=80
     DestructionEffect=None
     PhysicsType=OrientToFloor
     PhysicalRotationRate=(Pitch=16384,Yaw=16384,Roll=16384)
     AirTraction=1.000000
     bRestrictPitch=True
     MinPitch=-2048.000000
     maxPitch=2048.000000
     PitchUpRate=16384.000000
     PitchDownRate=16384.000000
     bRestrictRoll=True
     MinRoll=-2048.000000
     MaxRoll=2048.000000
     RollLeftRate=16384.000000
     RollRightRate=16384.000000
     YawLeftRate=2048.000000
     YawRightRate=2048.000000
     StationarySound=Sound'AmbModern.Looping.fan3'
     OperatingSound=Sound'AmbModern.Looping.fan3'
     CrashingSound=None
     DestroyedSound=None
     Elasticity=0.000000
     MaxSpeed=2048.000000
     MaxAcceleration=200.000000
     bRepairable=False
     MoverGlideType=MV_GlideByTime
     bDamageTriggered=True
}
