//=============================================================================
// DrivablePathMover.
// Moves between its pathnodes by means of its forward/backward control
// You can make a pathnode a child component of another drivable.
// Note that pathnodes need to have unique sequences within their tag, or the
// results are pretty unpredictable. Also, there is currently a maximum of 25
// nodes.
//=============================================================================
class DrivablePathMover expands drivable;

enum PathControlMethod {
 VehicleSpeed,
 VehicleTurn,
 TurretUp,
 TurretTurn,
 JumpDuck,
 FireAltFire
};

var DrivablePathNode pathnode[25];
var int pathnodecount;
var () name PathTag;
var () bool bLoopingPath;
var () bool bElasticPath;
var () PathControlMethod ControlMethod;
var DrivablePathNode currentnode;
var float currentposition;
var bool bPathOK;

simulated function PostBeginPlay()
{
 local DrivablePathNode prevnode,firstnode,pn;
 local int i;
 
 super.PostBeginPlay();
 
 Origin=vect(0,0,0);
 PlaneAngles=rot(0,0,0);
 
 currentposition=0;
 currentnode=None;
 
 foreach allactors(class'DrivablePathNode',pn,pathtag)
 {
  if(pn.sequence<arraycount(pathnode)) {
   pathnode[pn.sequence]=pn;                      // this sorts the path nodes
  }
 }
 prevnode=None;
 firstnode=None;
 
 for(i=0;i<arraycount(pathnode);i++) {            // assign previous and next 
  if(pathnode[i]!=None) {
   if(prevnode==None)                   // only on first pathnode encountered
    firstnode=pathnode[i];
   else 
    prevnode.Next=pathnode[i];
    
   pathnode[i].Prev=prevnode;
   prevnode=pathnode[i];
  }
 }
 if(prevnode!=None && firstnode!=prevnode) { // at least two pathnodes
  if(bLoopingPath) {
   prevnode.Next=firstNode;
   firstnode.prev=prevnode;
  }
  bPathOK=true;
  CurrentNode=firstnode;
  CurrentRotation=rotator(currentnode.next.location-currentnode.location);
  CurrentLocation=currentnode.location+vector(CurrentRotation)*currentposition;
  SetRotation(CurrentRotation);
  SetLocation(CurrentLocation);
 }
 else {
  bPathOK=false;
 }
 currentspeed=0;
}

simulated function InterpretControls(bool bJustFired,bool bFire,bool bJustAltFired,bool bAltFire,
                           float aVehicleSpeedControl,float aTurretTurn,float aVehicleTurn,
                           float aVehicleUp,float aTurretUp,bool bJustJumped,bool bJustDucked)
{
 local float newspeed;
 newspeed=0;
 
 switch(ControlMethod) {
  case VehicleSpeed:
       if(aVehicleSpeedControl>0)
        newspeed= (aVehicleSpeedControl/16384.0)*MaxSpeed;
       else 
        newspeed= abs(aVehicleSpeedControl/16384.0)*MinSpeed;
       break;
  case VehicleTurn:
       if(aVehicleTurn>0)
        newspeed= (aVehicleTurn/16384.0)*MaxSpeed;
       else 
        newspeed= abs(aVehicleTurn/16384.0)*MinSpeed;
       break;
  case TurretUp:
       if(aTurretUp>0)
        newspeed= (aTurretUp/16384.0)*MaxSpeed;
       else 
        newspeed= abs(aTurretUp/16384.0)*MinSpeed;
       break;
  case TurretTurn:
       if(aTurretTurn>0)
        newspeed= (aTurretTurn/16384.0)*MaxSpeed;
       else 
        newspeed= abs(aTurretTurn/16384.0)*MinSpeed;
       break;
  case JumpDuck:
       if(aVehicleUp>0)
        newspeed=MaxSpeed;
       else
       if(aVehicleUp<0)
        newspeed=MinSpeed;
       break;
  case FireAltFire:
       if(bFire)
        newspeed=MaxSpeed;
       else
       if(bAltFire)
        newspeed=MinSpeed;
       break;
 }
 desiredspeed=newspeed;
}

simulated function DoPhysics(float DeltaTime)
{
 local float accel;
 local float PreviousPosition;
 local DrivablePathNode PreviousNode;
 
 if(bPathOK) {
  PreviousPosition=CurrentPosition;
  PreviousNode    =CurrentNode;
  PreviousRotation=CurrentRotation;
  PreviousLocation=CurrentLocation;

  currentposition+=currentspeed*deltatime;
  checknewposition();
 
  accel=0.0;
  if(MaxSpeed>0) {
   if(DesiredSpeed<CurrentSpeed)
    accel=MAX(DesiredSpeed-CurrentSpeed,-MaxDeceleration*deltatime); // note: desiredspeed-currentspeed is negative, so accel is negative 

   if(DesiredSpeed>CurrentSpeed)
    accel=MIN(DesiredSpeed-CurrentSpeed,MaxAcceleration*deltatime); 
  }
  CurrentSpeed+=accel;
  if(CurrentSpeed>MaxSpeed)
   CurrentSpeed=MaxSpeed;
  else
  if(CurrentSpeed<MinSpeed)
   CurrentSpeed=MinSpeed;

  linearvelocity=vector(rotator(currentnode.next.location-currentnode.location))*currentspeed;
  CurrentRotation=rotator(currentnode.next.location-currentnode.location);
  CurrentLocation=currentnode.location+vector(CurrentRotation)*currentposition;
 
  if(CheckForCollision(deltatime)) {
   CurrentNode=PreviousNode;
   CurrentPosition=PreviousPosition;
   CurrentSpeed=0;
  }
 }
}

// CheckNewPosition updates the currentposition and currentnode 
// this will temporarily 'halt' at the node so DoPhysics can use a linear CheckForCollision
simulated function CheckNewPosition()     
{
 local bool c;

 if(currentposition>0 && currentnode.next==None) {
  currentposition=0;
  currentspeed=0;
  return;
 }
 
 c=true;
 while(c) {
  c=false;
  if(currentposition<0) {
   if(currentnode.prev!=None) { // travel to previous stretch
    c=true;
    currentnode=currentnode.prev;
    currentposition=vsize(currentnode.location-currentnode.next.location);
   }
   else {
    currentposition=0;
    currentspeed=0;
   }
  }
  else
  if(currentposition>vsize(currentnode.location-currentnode.next.location)) {
   if(currentnode.next!=None) { // travel to next stretch
    c=true;
    currentposition=0;
    currentnode=currentnode.next;
    if(currentposition>0 && currentnode.next==None) {
     currentposition=0;
     currentspeed=0;
     return;
    }
   }
   else {
    currentposition=vsize(currentnode.location-currentnode.next.location);
    currentspeed=0;
   }
  }
 }
}

simulated function RotationPhysics(float DeltaTime)
{
}

defaultproperties
{
     PhysicalRotationRate=(Pitch=0,Yaw=0,Roll=0)
     CrashingSound=None
     DestroyedSound=None
     bRepairable=False
     RepairTime=0.000000
     MoverGlideType=MV_GlideByTime
     bDamageTriggered=True
}
