//=============================================================================
// DrivableLadder.
//=============================================================================
// Special ladder type object that can be moved and rotated with a drivable.
// on this ladder you want to look up in order to climb up, look down to go down
// When you jump, you let go of the ladder...
class DrivableLadder expands Pickup;
var DrivableLadder Original;               // set in SpawnCopy
var bool go;
var float aircontrol;
var float ownerspeed;
var bool ownerwantstomove;

final function int sign(float value)
{
 if(value==0)
  return 0;
 if(value<0)
  return -1;
 return 1;
}


// atan2 determines 360 degree angle (between -pi and Pi, more precisely)
final function float atan2(float Y,float X)
{
 local float tempang;
 
 if(X==0) {
  if(Y<0)
   tempang=-pi/2;
  else if(Y>0)
   tempang=pi/2;
  else
   tempang=0;
 }
 else if(X<0) {
  tempang=atan(Y/X)+pi; // Third quadrant
 }
 else 
  tempang=atan(Y/X);        // 
   
 if(tempang>pi) 
  tempang-=pi*2;
 
 if(tempang<-pi)
  tempang+=pi*2;
   
 return tempang;
}

// Possibly useful functions below

// RealWorldLocation determines the position of a point after its origin is rotated.
// this is needed to properly rotate and move a child when its starting rotation is different
// from that of the parent
// (otherwise, the 'rotateby' could just be added to the child's rotation)
final function vector RealWorldLocation(vector oldpoint,rotator rotateby)
{
 local vector temppoint;
 local rotator temprot;
 local float dist;
 local float axisdist;
 local float axisangle;
 
 temppoint=oldpoint;
 dist = VSize(temppoint);
 temppoint = temppoint/dist; // unity vector, requires less calculations
 
// rotate around X-axis (Roll)
 axisangle = atan2(temppoint.Z,temppoint.Y);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.Y));
 temppoint.Z= sin(axisangle-(RotateBy.Roll*pi)/32768.0)*axisdist;
 temppoint.Y= cos(axisangle-(RotateBy.Roll*pi)/32768.0)*axisdist;

 
// rotate around Y-axis (Pitch)
 axisangle = atan2(temppoint.Z,temppoint.X);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.X));
 temppoint.Z= sin(axisangle+(RotateBy.Pitch*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle+(RotateBy.Pitch*pi)/32768.0)*axisdist;

 temprot=rotateby;
 temprot.Roll=0;
 temprot.Pitch=0;
// return origin+vector(rotator(temppoint)+temprot)*dist;
 
// rotate around Z-axis (Yaw) 
 axisangle = atan2(temppoint.Y,temppoint.X);
 axisdist  = sqrt(square(temppoint.Y)+square(temppoint.X));
 temppoint.Y= sin(axisangle+(RotateBy.Yaw*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle+(RotateBy.Yaw*pi)/32768.0)*axisdist;
 
 return vector(rotator(temppoint))*dist; 
}

final function vector VirtualLocation(vector oldpoint,rotator rotateby)
{
 local vector temppoint;
 local rotator temprot;
 local float dist;
 local float axisdist;
 local float axisangle;
 
 temppoint=oldpoint;
 dist = VSize(temppoint);
 temppoint = temppoint/dist; // unity vector, requires less calculations
 
// rotate around Z-axis (Yaw) 
 axisangle = atan2(temppoint.Y,temppoint.X);
 axisdist  = sqrt(square(temppoint.Y)+square(temppoint.X));
 temppoint.Y= sin(axisangle-(RotateBy.Yaw*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle-(RotateBy.Yaw*pi)/32768.0)*axisdist;
 
// rotate around Y-axis (Pitch)
 axisangle = atan2(temppoint.Z,temppoint.X);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.X));
 temppoint.Z= sin(axisangle-(RotateBy.Pitch*pi)/32768.0)*axisdist;
 temppoint.X= cos(axisangle-(RotateBy.Pitch*pi)/32768.0)*axisdist;

// rotate around X-axis (Roll)
 axisangle = atan2(temppoint.Z,temppoint.Y);
 axisdist  = sqrt(square(temppoint.Z)+square(temppoint.Y));
 temppoint.Z= sin(axisangle+(RotateBy.Roll*pi)/32768.0)*axisdist;
 temppoint.Y= cos(axisangle+(RotateBy.Roll*pi)/32768.0)*axisdist;
 
 temprot=rotateby;
 temprot.Roll=0;
 temprot.Pitch=0;
// return origin+vector(rotator(temppoint)+temprot)*dist;
 
 return vector(rotator(temppoint))*dist; // add origin and distance again
}

// VirtualRotation rotates a rotation
final function rotator VirtualRotation(rotator CurrentRotation,rotator planeangles)
{
 local vector X,Y;
 
 //return CurrentRotation+planeangles;
 
 X=RealWorldLocation(vect(1,0,0),CurrentRotation);
 Y=RealWorldLocation(vect(0,1,0),CurrentRotation);
 X=VirtualLocation(X,planeangles);
 Y=VirtualLocation(Y,planeangles);
 
 return GetPlaneRotation(X,Y);
}

// getplanerotation determines the angles of a plane defined by two vectors, relative to the ground.
// Two vectors are needed, because roll needs to be determined.
final function rotator GetPlaneRotation(vector Xaxis,vector Yaxis)
{
 local rotator temprot;
 local rotator rotat;
 local vector X,Y;

 X=Xaxis/VSize(Xaxis);
 Y=Yaxis/VSize(Yaxis);

 rotat.Yaw=(atan2(X.y,X.x)*32768.0)/pi;
// rotat.Yaw=rotator(X).Yaw;
 temprot = rot(0,0,0);
 temprot.Yaw=-rotat.Yaw;
 X = RealWorldLocation(X,temprot);
 Y = RealWorldLocation(Y,temprot);
 rotat.Pitch = (atan2(X.z,X.x)*32768.0)/pi;
// rotat.Pitch=rotator(X).Pitch;
 temprot = rot(0,0,0);
 temprot.Pitch=-rotat.Pitch;
 X = RealWorldLocation(X,temprot);
 Y = RealWorldLocation(Y,temprot);
 rotat.Roll = -(atan2(Y.Z,Y.Y)*32768.0)/pi;               
 temprot = rot(0,0,0);
 temprot.Roll=-rotat.Roll;
 X = RealWorldLocation(X,temprot);
 Y = RealWorldLocation(Y,temprot);
  
 return rotat;
}

// RealWorldRotation rotates a rotation
final function rotator RealWorldRotation(rotator CurrentRotation,rotator planeangles)
{
 local vector X,Y;
 
 //return CurrentRotation+planeangles;
 
 X=RealWorldLocation(vect(1,0,0),CurrentRotation);
 Y=RealWorldLocation(vect(0,1,0),CurrentRotation);
 X=RealWorldLocation(X,planeangles);
 Y=RealWorldLocation(Y,planeangles);
 
 return GetPlaneRotation(X,Y);
}

final function rotator InverseRotation(rotator rotat)
{
 local rotator newrot;

 newrot.Yaw=-rotat.Yaw;
 newrot.Pitch=-rotat.Pitch;
 newrot.Roll=-rotat.Roll;
 return newrot;
}

function bool LadderCheck(pawn specificuser)
{
 local pawn user;
 local float userrot;

 foreach touchingactors(class'pawn',user) {
  if(user==specificuser) {
   userrot=VirtualRotation(user.rotation,rotation).Yaw;
   if(userrot>-16384 && userrot<16384)
    return true;
   else
    return false;
  }
 }
 return false;
}

function Tick(float DeltaTime)
{
 local pawn p;
 
 if(Owner==None) { // only pickup item, not inventory item
  foreach touchingactors(class'pawn',p) {
   if(p.FindInventoryType(class'drivableladder')==None)
    SpawnCopy(p);
  }
 }
 else
 if(go && Original!=None) { 
  p=pawn(owner);
  if(Original.LadderCheck(p)) {
   p.setphysics(PHYS_Flying);
   p.TweenToRunning(deltatime);
   ownerspeed=p.groundspeed/2*(virtuallocation(vector(p.viewrotation),original.rotation).Z);
   p.velocity=ownerspeed*realworldlocation(vect(0,0,1),original.rotation);//-owner.region.zone.zonegravity*deltatime; // any player can take two steps per second 
  }
  else {
   p.setphysics(PHYS_Falling);
   Destroy();
  }
 }
}

function inventory SpawnCopy( pawn Other )
{
	local DrivableLadder Copy;

	Copy = spawn(Class,Other);
	Copy.Tag           = Tag;
	Copy.Event         = Event;
	Copy.Original=Self;

	Copy.RespawnTime = 0.0;
	Copy.aircontrol=other.aircontrol;
	Copy.GiveTo( Other );
	copy.ownerwantstomove=false;
	Copy.Go = true;
	Copy.SetTimer(1,true);

//	log("Pawn touched me");
	return Copy;
}

function SetRespawn()
{
}

function bool validtouch(actor other)
{
 return true;
}

function OwnerJumped()
{ 
 ownerwantstomove=!ownerwantstomove;
}

function BeginPlay()
{
 super.beginplay();
 DrawType=DT_None;
}

defaultproperties
{
     bAutoActivate=True
     PickupMessage=""
     BobDamping=1.000000
     MaxDesireability=0.000000
     M_Activated=""
     M_Selected=""
     M_Deactivated=""
     bHidden=True
     DrawType=DT_Sprite
     Texture=Texture'Editor.B_MenuUp'
     CollisionRadius=20.000000
     CollisionHeight=128.000000
}
